# -*- coding: utf-8 -*-
#
# Copyright (c) 2011-2013, Cédric Krier
# Copyright (c) 2011-2013, B2CK
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#     * Neither the name of the <organization> nor the
#       names of its contributors may be used to endorse or promote products
#       derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import unittest

from sql import Table, Flavor
from sql.functions import Function, Abs, Overlay, Trim, AtTimeZone


class TestFunctions(unittest.TestCase):
    table = Table('t')

    def test_abs(self):
        abs_ = Abs(self.table.c1)
        self.assertEqual(str(abs_), 'ABS("c1")')
        self.assertEqual(abs_.params, ())

        abs_ = Abs(-12)
        self.assertEqual(str(abs_), 'ABS(%s)')
        self.assertEqual(abs_.params, (-12,))

    def test_mapping(self):
        class MyAbs(Function):
            _function = 'MY_ABS'
            params = ('test',)

        abs_ = Abs(self.table.c1)
        flavor = Flavor(function_mapping={
                Abs: MyAbs,
                })
        Flavor.set(flavor)
        try:
            self.assertEqual(str(abs_), 'MY_ABS("c1")')
            self.assertEqual(abs_.params, ('test',))
        finally:
            Flavor.set(Flavor())

    def test_overlay(self):
        overlay = Overlay(self.table.c1, 'test', 3)
        self.assertEqual(str(overlay), 'OVERLAY("c1" PLACING %s FROM %s)')
        self.assertEqual(overlay.params, ('test', 3))
        overlay = Overlay(self.table.c1, 'test', 3, 7)
        self.assertEqual(str(overlay),
            'OVERLAY("c1" PLACING %s FROM %s FOR %s)')
        self.assertEqual(overlay.params, ('test', 3, 7))

    def test_trim(self):
        trim = Trim(' test ')
        self.assertEqual(str(trim), 'TRIM(BOTH %s FROM %s)')
        self.assertEqual(trim.params, (' ', ' test ',))

    def test_at_time_zone(self):
        time_zone = AtTimeZone(self.table.c1, 'UTC')
        self.assertEqual(str(time_zone), '"c1" AT TIME ZONE %s')
        self.assertEqual(time_zone.params, ('UTC',))
