/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var objectKeys = require( './../../keys' );
var isPlainObject = require( '@stdlib/assert/is-plain-object' );
var isArray = require( '@stdlib/assert/is-array' );


// MAIN //

/**
* Recursively flattens an object.
*
* @private
* @param {Object} out - output object
* @param {ObjectLike} obj - input object
* @param {string} prefix - key prefix
* @param {NonNegativeInteger} depth - recursion depth
* @param {Options} opts - options
* @param {boolean} opts.flattenArrays - boolean indicating whether to flatten arrays
* @param {string} opts.delimiter - key path delimiter
* @returns {Object} output object
*/
function recurse( out, obj, prefix, depth, opts ) {
	var keys;
	var val;
	var key;
	var i;
	if ( prefix ) {
		prefix += opts.delimiter;
	}
	keys = objectKeys( obj );
	for ( i = 0; i < keys.length; i++ ) {
		val = obj[ keys[i] ];
		key = prefix + keys[ i ];
		if ( depth ) {
			if (
				(isPlainObject( val ) && objectKeys( val ).length) ||
				(opts.flattenArrays && isArray( val ))
			) {
				recurse( out, val, key, depth-1, opts );
				continue;
			}
		}
		out[ key ] = val;
	}
	return out;
}


// EXPORTS //

module.exports = recurse;
