/* $Id: kmo_illumination-test.c,v 1.17 2013-10-08 11:18:56 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:56 $
 * $Revision: 1.17 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>

#include <cpl.h>

#include "kmclipm_functions.h"

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

const char      *path_recipe        = "kmo_illumination/";

const char      *valid_files[]      = {"sky_123.fits",
                                       "sky_231.fits",
                                       "sky_312.fits"};

/**
    @defgroup kmo_illumination   kmo_illumination unit tests
*/

/**
  @brief    Test of kmo_illumination recipe.
  @param    argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_illumination-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_illumination-test @c --no-gen when only the tests should be
  executed and no data should be generated.
 */
int main(int argc, char *argv[])
{
    float   tol     = 0.01;

    char    test_path[256],
            file_path[256],
            op1_file[256],
            op2_file[256],
            op3_file[256],
            esorex_command[512];

    const char *sof_path = "test_illumination.sof";

    cpl_propertylist *h = NULL;

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    cpl_msg_info(cpl_func, "Testing kmo_illumination with real "
                           "pipeline data.");
    sprintf(test_path, "%s%s", test_global_path_test_data, "pipeline/");

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    // get rid of warning...
    if ((argc == 5) && (strcmp(argv[1], "xxx") != 0)) {
    }

    FILE *fd = fopen("log_kmo_illumination.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    //
    //  PIPELINE TEST (must be called after kmo_wave_cal-test, since the
    //  following recipes rely on this output)
    //
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[1]);
    kmo_test_cat_strings(op3_file, test_path, valid_files[2]);

    // create sof-file
    FILE *fh = fopen(sof_path, "w");
    char *my_path = cpl_sprintf("%s/ref_data/kmos_wave_band.fits", getenv("srcdir"));
    fprintf (fh,
             "%s                           FLAT_SKY\n"
             "%s                           FLAT_SKY\n"
             "%s                           FLAT_SKY\n"
             "master_dark.fits             MASTER_DARK\n"
             "master_flat_KKK.fits         MASTER_FLAT\n"
             "xcal_KKK.fits                XCAL\n"
             "ycal_KKK.fits                YCAL\n"
             "lcal_KKK.fits                LCAL\n"
             "%s WAVE_BAND\n",
             op1_file, op2_file, op3_file, my_path);
    fclose(fh);
    cpl_free(my_path);

    // create esorex-command
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_illumination "
            "--add-all --imethod=NN --b_start=1.948 --b_end=2.351 %s", sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_illumination.txt", sof_path, TRUE);

    // execute esorex-command
    cpl_test_eq(0, system(esorex_command));

    // checking output
    strcpy(file_path, "illum_corr_KKK");
    cpl_test_abs(kmo_test_esorex_data(file_path, 1), 24.7136, tol);
    strcat(file_path, ".fits");

    h = kmclipm_propertylist_load(file_path, 0);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_SPAT_UNIF),
                 0.00265038, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_SPAT_MAX_DEV),
                 0.00448508 , tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_SPAT_MAX_NONUNIF),
                 0.0332622, tol);
    cpl_propertylist_delete(h); h = NULL;

    return cpl_test_end(0);
}

/** @} */
