// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/first_run/goodies_displayer.h"

#include "base/prefs/pref_service.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/chromeos/login/startup_utils.h"
#include "chrome/browser/chromeos/policy/browser_policy_connector_chromeos.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/browser/ui/browser_list.h"
#include "chrome/browser/ui/browser_tabstrip.h"
#include "chrome/common/pref_names.h"
#include "content/public/browser/browser_thread.h"

namespace chromeos {
namespace first_run {

namespace {

GoodiesDisplayer* g_goodies_displayer = nullptr;
GoodiesDisplayerTestInfo* g_test_info = nullptr;

// Checks timestamp on OOBE Complete flag file, or use fake device age for test.
// kCanShowOobeGoodiesPage defaults to |true|; set to |false| (return |false|)
// for any device over kMaxDaysAfterOobeForGoodies days old, to avoid showing
// Goodies after update on older devices.
bool CheckGoodiesPrefAgainstOobeTimestamp() {
  DCHECK(content::BrowserThread::GetBlockingPool()->RunsTasksOnCurrentThread());
  const int days_since_oobe =
      g_test_info ? g_test_info->days_since_oobe
                  : StartupUtils::GetTimeSinceOobeFlagFileCreation().InDays();
  return days_since_oobe <= GoodiesDisplayer::kMaxDaysAfterOobeForGoodies;
}

// Callback into main thread to set pref to |false| if too long since oobe, or
// to create GoodiesDisplayer otherwise.
void UpdateGoodiesPrefCantShow(bool can_show_goodies) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);
  if (can_show_goodies) {
    if (!g_goodies_displayer)
      g_goodies_displayer = new GoodiesDisplayer();
  } else {
    g_browser_process->local_state()->SetBoolean(prefs::kCanShowOobeGoodiesPage,
                                                 false);
  }

  if (g_test_info) {
    g_test_info->setup_complete = true;
    if (!g_test_info->on_setup_complete_callback.is_null())
      g_test_info->on_setup_complete_callback.Run();
  }
}

}  // namespace

const char GoodiesDisplayer::kGoodiesURL[] =
    "https://www.google.com/chrome/devices/goodies.html";

GoodiesDisplayer::GoodiesDisplayer() {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);
  BrowserList::AddObserver(this);
}

GoodiesDisplayer::~GoodiesDisplayer() {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);
  BrowserList::RemoveObserver(this);
}

// If Goodies page hasn't been shown yet, and Chromebook isn't too old, create
// GoodiesDisplayer to observe BrowserList.  Return |true| if checking age.
// static
bool GoodiesDisplayer::Init() {
  const bool can_show = g_browser_process->local_state()->GetBoolean(
      prefs::kCanShowOobeGoodiesPage);
  if (can_show) {
    base::PostTaskAndReplyWithResult(
        content::BrowserThread::GetBlockingPool(), FROM_HERE,
        base::Bind(&CheckGoodiesPrefAgainstOobeTimestamp),
        base::Bind(&UpdateGoodiesPrefCantShow));
  }
  return can_show;
}

// static
void GoodiesDisplayer::InitForTesting(GoodiesDisplayerTestInfo* test_info) {
  CHECK(!g_test_info) << "GoodiesDisplayer::InitForTesting called twice";
  g_test_info = test_info;
  test_info->setup_complete = !Init();
}

// static
void GoodiesDisplayer::Delete() {
  delete g_goodies_displayer;
  g_goodies_displayer = nullptr;
}

// If conditions enumerated below are met, this loads the Oobe Goodies page for
// new Chromebooks; when appropriate, it uses pref to mark page as shown,
// removes itself from BrowserListObservers, and deletes itself.
void GoodiesDisplayer::OnBrowserSetLastActive(Browser* browser) {
  // 1. Must be an actual tabbed brower window.
  if (browser->type() != Browser::TYPE_TABBED)
    return;

  // 2. Not guest or incognito session (keep observing).
  if (browser->profile()->IsOffTheRecord())
    return;

  PrefService* local_state = g_browser_process->local_state();
  // 3. Not previously shown, or otherwise marked as unavailable.
  if (local_state->GetBoolean(prefs::kCanShowOobeGoodiesPage)) {
    // 4. Device not enterprise enrolled.
    if (!g_browser_process->platform_part()
             ->browser_policy_connector_chromeos()
             ->IsEnterpriseManaged())
      chrome::AddTabAt(browser, GURL(kGoodiesURL), 2, false);

    // Set to |false| whether enterprise enrolled or Goodies shown.
    local_state->SetBoolean(prefs::kCanShowOobeGoodiesPage, false);
  }

  // Regardless of how we got here, we don't henceforth need to show Goodies.
  base::MessageLoop::current()->PostTask(FROM_HERE, base::Bind(&Delete));
}

GoodiesDisplayerTestInfo::GoodiesDisplayerTestInfo()
    : days_since_oobe(0), setup_complete(false) {}

GoodiesDisplayerTestInfo::~GoodiesDisplayerTestInfo() {}

}  // namespace first_run
}  // namespace chromeos
