// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_EXTENSIONS_EXTENSION_FUNCTION_REGISTRY_H_
#define CHROME_BROWSER_EXTENSIONS_EXTENSION_FUNCTION_REGISTRY_H_

#include <map>
#include <string>
#include <vector>

#include "chrome/browser/extensions/extension_function_histogram_value.h"

class ExtensionFunction;

// A factory function for creating new ExtensionFunction instances.
typedef ExtensionFunction* (*ExtensionFunctionFactory)();

// Template for defining ExtensionFunctionFactory.
template<class T>
ExtensionFunction* NewExtensionFunction() {
  return new T();
}

// Contains a list of all known extension functions and allows clients to
// create instances of them.
class ExtensionFunctionRegistry {
 public:
  static ExtensionFunctionRegistry* GetInstance();
  explicit ExtensionFunctionRegistry();
  virtual ~ExtensionFunctionRegistry();

  // Resets all functions to their default values.
  void ResetFunctions();

  // Adds all function names to 'names'.
  void GetAllNames(std::vector<std::string>* names);

  // Allows overriding of specific functions (e.g. for testing).  Functions
  // must be previously registered.  Returns true if successful.
  bool OverrideFunction(const std::string& name,
                        ExtensionFunctionFactory factory);

  // Factory method for the ExtensionFunction registered as 'name'.
  ExtensionFunction* NewFunction(const std::string& name);

  template<class T>
  void RegisterFunction() {
    ExtensionFunctionFactory factory = &NewExtensionFunction<T>;
    factories_[T::function_name()] =
        FactoryEntry(factory, T::histogram_value());
  }

  struct FactoryEntry {
   public:
    explicit FactoryEntry();
    explicit FactoryEntry(ExtensionFunctionFactory factory,
      extensions::functions::HistogramValue histogram_value);

    ExtensionFunctionFactory factory_;
    extensions::functions::HistogramValue histogram_value_;
  };

  typedef std::map<std::string, FactoryEntry> FactoryMap;
  FactoryMap factories_;
};

#endif  // CHROME_BROWSER_EXTENSIONS_EXTENSION_FUNCTION_REGISTRY_H_
