//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Detector/DetectorItem.cpp
//! @brief     Implements classes DetectorItems
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Detector/DetectorItem.h"
#include "Device/Detector/IDetector.h"
#include "Device/Mask/IShape2D.h"
#include "Device/Resolution/ResolutionFunction2DGaussian.h"
#include "GUI/Model/Detector/ResolutionFunctionItems.h"
namespace {
namespace Tag {

const QString ResolutionFunction("ResolutionFunction");
const QString ExpandResolutionFuncGroupbox("ExpandGroupbox");

} // namespace Tag
} // namespace

DetectorItem::DetectorItem() = default;

DetectorItem::~DetectorItem() = default;

std::unique_ptr<IDetector> DetectorItem::createDetector() const
{
    auto result = createDomainDetector();
    addMasksToCore(result.get());

    if (auto resFunc = createResolutionFunction())
        result->setResolutionFunction(*resFunc);

    return result;
}

void DetectorItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // resolution function
    w->writeStartElement(Tag::ResolutionFunction);
    m_resolutionFunction.writeTo(w);
    w->writeEndElement();

    // resolution function groupbox: is expanded?
    w->writeStartElement(Tag::ExpandResolutionFuncGroupbox);
    XML::writeAttribute(w, XML::Attrib::value, m_expandResolutionFunc);
    w->writeEndElement();
}

void DetectorItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // resolution function
        if (tag == Tag::ResolutionFunction) {
            m_resolutionFunction.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // resolution function groupbox: is expanded?
        } else if (tag == Tag::ExpandResolutionFuncGroupbox) {
            XML::readAttribute(r, XML::Attrib::value, &m_expandResolutionFunc);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

void DetectorItem::importMasks(const MaskContainerItem* maskContainer)
{
    m_maskContainerItem.copyFrom(maskContainer);
}

MaskContainerItem& DetectorItem::maskItems()
{
    return m_maskContainerItem;
}

SelectionProperty<ResolutionFunctionItemCatalog>& DetectorItem::resolutionFunctionSelection()
{
    return m_resolutionFunction;
}

std::unique_ptr<IResolutionFunction2D> DetectorItem::createResolutionFunction() const
{
    return m_resolutionFunction->createResolutionFunction(axesToCoreUnitsFactor());
}

void DetectorItem::addMasksToCore(IDetector* detector) const
{
    const double scale = axesToCoreUnitsFactor();
    const QVector<MaskItem*> maskItems = m_maskContainerItem.maskItems();
    for (auto maskIter = maskItems.rbegin(); maskIter != maskItems.rend(); maskIter++)
        if ((*maskIter)->isVisible()) {
            if (auto* roiItem = dynamic_cast<RegionOfInterestItem*>(*maskIter)) {
                const double xlow = scale * roiItem->xLow();
                const double ylow = scale * roiItem->yLow();
                const double xup = scale * roiItem->xUp();
                const double yup = scale * roiItem->yUp();
                detector->setRegionOfInterest(xlow, ylow, xup, yup);
            } else {
                std::unique_ptr<IShape2D> shape((*maskIter)->createShape(scale));
                detector->addMask(*shape, (*maskIter)->maskValue());
            }
        }
}
