/*
 * Copyright (c) 2019-2021, 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <cstdint>

namespace arm_gemm {

void a64_hybrid_s8qs_dot_6x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int col_base
)
{
    struct KernelArgs {
        const int32_t *multiplier_ptr = {};
        const int32_t *shift_ptr = {};
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
    } ka;

    unsigned long flags=0;
    void *output_ptr;
    void *input_ptr;

    if (output_arg.is_indirect) {
        output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    if (qp->per_channel_requant) {
        flags |= 0x10;
        ka.multiplier_ptr=qp->per_channel_muls + col_base;
        ka.shift_ptr=qp->per_channel_right_shifts + col_base;
    }
    if (qp->c_offset > qp->minval) {
        flags |= 0x20;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 136f\n"
      "cmp %x[M], #0x4\n"
      "bgt 109f\n"
      "beq 82f\n"
      "cmp %x[M], #0x2\n"
      "bgt 55f\n"
      "beq 28f\n"
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "2:"  // Height 1: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "3:"  // Height 1: setup done
      "mov x28, #0x0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x27, #0x10\n"
      "blt 9f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q6, [x9, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q7, [x9, #0x10]\n"
      "blt 8f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4f80e22a  // sdot v10.4s, v17.16b, v0.4b[0]\n"
      "ldr q17, [x9, #0x40]\n"
      ".inst 0x4f80e20b  // sdot v11.4s, v16.16b, v0.4b[0]\n"
      "ldr q16, [x9, #0x50]\n"
      ".inst 0x4fa0e228  // sdot v8.4s, v17.16b, v0.4b[1]\n"
      "ldr q17, [x9, #0x60]\n"
      ".inst 0x4fa0e209  // sdot v9.4s, v16.16b, v0.4b[1]\n"
      "ldr q16, [x9, #0x70]\n"
      ".inst 0x4fa0e22a  // sdot v10.4s, v17.16b, v0.4b[1]\n"
      "ldr q17, [x9, #0x80]\n"
      ".inst 0x4fa0e20b  // sdot v11.4s, v16.16b, v0.4b[1]\n"
      "ldr q16, [x9, #0x90]\n"
      ".inst 0x4f80ea28  // sdot v8.4s, v17.16b, v0.4b[2]\n"
      "ldr q17, [x9, #0xa0]\n"
      ".inst 0x4f80ea09  // sdot v9.4s, v16.16b, v0.4b[2]\n"
      "ldr q16, [x9, #0xb0]\n"
      ".inst 0x4f80ea2a  // sdot v10.4s, v17.16b, v0.4b[2]\n"
      "ldr q17, [x9, #0xc0]\n"
      ".inst 0x4f80ea0b  // sdot v11.4s, v16.16b, v0.4b[2]\n"
      "ldr q16, [x9, #0xd0]\n"
      ".inst 0x4fa0ea28  // sdot v8.4s, v17.16b, v0.4b[3]\n"
      "ldr q17, [x9, #0xe0]\n"
      ".inst 0x4fa0ea09  // sdot v9.4s, v16.16b, v0.4b[3]\n"
      "ldr q16, [x9, #0xf0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4fa0ea2a  // sdot v10.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x4fa0ea0b  // sdot v11.4s, v16.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      "cmp x27, #0x20\n"
      "add x9, x9, #0x100\n"
      "ldr q6, [x9, #0x0]\n"
      "ldr q7, [x9, #0x10]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "bge 7b\n"
      "8:"  // Height 1: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4f80e22a  // sdot v10.4s, v17.16b, v0.4b[0]\n"
      "ldr q17, [x9, #0x40]\n"
      ".inst 0x4f80e20b  // sdot v11.4s, v16.16b, v0.4b[0]\n"
      "ldr q16, [x9, #0x50]\n"
      ".inst 0x4fa0e228  // sdot v8.4s, v17.16b, v0.4b[1]\n"
      "ldr q17, [x9, #0x60]\n"
      ".inst 0x4fa0e209  // sdot v9.4s, v16.16b, v0.4b[1]\n"
      "ldr q16, [x9, #0x70]\n"
      ".inst 0x4fa0e22a  // sdot v10.4s, v17.16b, v0.4b[1]\n"
      "ldr q17, [x9, #0x80]\n"
      ".inst 0x4fa0e20b  // sdot v11.4s, v16.16b, v0.4b[1]\n"
      "ldr q16, [x9, #0x90]\n"
      ".inst 0x4f80ea28  // sdot v8.4s, v17.16b, v0.4b[2]\n"
      "ldr q17, [x9, #0xa0]\n"
      ".inst 0x4f80ea09  // sdot v9.4s, v16.16b, v0.4b[2]\n"
      "ldr q16, [x9, #0xb0]\n"
      ".inst 0x4f80ea2a  // sdot v10.4s, v17.16b, v0.4b[2]\n"
      "ldr q17, [x9, #0xc0]\n"
      ".inst 0x4f80ea0b  // sdot v11.4s, v16.16b, v0.4b[2]\n"
      "ldr q16, [x9, #0xd0]\n"
      ".inst 0x4fa0ea28  // sdot v8.4s, v17.16b, v0.4b[3]\n"
      "ldr q17, [x9, #0xe0]\n"
      ".inst 0x4fa0ea09  // sdot v9.4s, v16.16b, v0.4b[3]\n"
      "ldr q16, [x9, #0xf0]\n"
      "add x26, x26, #0x10\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4fa0ea2a  // sdot v10.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x4fa0ea0b  // sdot v11.4s, v16.16b, v0.4b[3]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "add x9, x9, #0x100\n"
      "9:"  // Height 1: Multiply loop: Main loop skip
      "cbz x27, 14f\n"
      "cmp x27, #0x4\n"
      "blt 11f\n"
      "10:"  // Height 1: Multiply loop: Odd block loop
      "ldr s18, [x26], #0x4\n"
      "ldr q16, [x9, #0x0]\n"
      ".inst 0x4f92e208  // sdot v8.4s, v16.16b, v18.4b[0]\n"
      "sub x27, x27, #0x4\n"
      "ldr q16, [x9, #0x10]\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4f92e209  // sdot v9.4s, v16.16b, v18.4b[0]\n"
      "cmp x27, #0x4\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4f92e22a  // sdot v10.4s, v17.16b, v18.4b[0]\n"
      ".inst 0x4f92e20b  // sdot v11.4s, v16.16b, v18.4b[0]\n"
      "add x9, x9, #0x40\n"
      "bge 10b\n"
      "11:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x27, 14f\n"
      "tbz x27, #1, 12f\n"
      "ldr h0, [x26], #0x2\n"
      "tbz x27, #0, 13f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "b 13f\n"
      "12:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "13:"  // Height 1: Multiply loop: Ragged operand read: Done
      "ldr q17, [x9, #0x0]\n"
      "ldr q16, [x9, #0x10]\n"
      ".inst 0x4f80e228  // sdot v8.4s, v17.16b, v0.4b[0]\n"
      ".inst 0x4f80e209  // sdot v9.4s, v16.16b, v0.4b[0]\n"
      "ldr q17, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4f80e22a  // sdot v10.4s, v17.16b, v0.4b[0]\n"
      ".inst 0x4f80e20b  // sdot v11.4s, v16.16b, v0.4b[0]\n"
      "add x9, x9, #0x40\n"
      "14:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 4b\n"
      "ldr q17, [x14, #0x0]\n"
      "ldr q16, [x14, #0x10]\n"
      "add v8.4s, v8.4s, v17.4s\n"
      "add v9.4s, v9.4s, v16.4s\n"
      "ldr q17, [x14, #0x20]\n"
      "ldr q16, [x14, #0x30]\n"
      "add v10.4s, v10.4s, v17.4s\n"
      "add v11.4s, v11.4s, v16.4s\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "add x14, x14, #0x40\n"
      "tbz %x[flags], #4, 15f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 16f\n"
      "15:"  // Height 1: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "16:"  // Height 1: parameters loaded
      "sqrdmulh v8.4s, v8.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "tbz %x[flags], #5, 17f\n"
      "and v19.16b, v8.16b, v0.16b\n"
      "and v18.16b, v9.16b, v1.16b\n"
      "and v17.16b, v10.16b, v2.16b\n"
      "and v16.16b, v11.16b, v3.16b\n"
      "sshr v19.4s, v19.4s, #0x1f\n"
      "sshr v18.4s, v18.4s, #0x1f\n"
      "sshr v17.4s, v17.4s, #0x1f\n"
      "sshr v16.4s, v16.4s, #0x1f\n"
      "sqadd v8.4s, v8.4s, v19.4s\n"
      "sqadd v9.4s, v9.4s, v18.4s\n"
      "sqadd v10.4s, v10.4s, v17.4s\n"
      "sqadd v11.4s, v11.4s, v16.4s\n"
      "17:"  // Height 1: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v18.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v17.4s }, [x20]\n"
      "add v8.4s, v8.4s, v18.4s\n"
      "add v9.4s, v9.4s, v18.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v16.4s }, [x20]\n"
      "add v10.4s, v10.4s, v18.4s\n"
      "add v11.4s, v11.4s, v18.4s\n"
      "cmp x10, #0x10\n"
      "smin v8.4s, v8.4s, v17.4s\n"
      "smin v9.4s, v9.4s, v17.4s\n"
      "smin v10.4s, v10.4s, v17.4s\n"
      "smin v11.4s, v11.4s, v17.4s\n"
      "smax v8.4s, v8.4s, v16.4s\n"
      "smax v9.4s, v9.4s, v16.4s\n"
      "smax v10.4s, v10.4s, v16.4s\n"
      "smax v11.4s, v11.4s, v16.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v16.8h, v10.8h, v11.8h\n"
      "uzp1 v8.16b, v8.16b, v16.16b\n"
      "bge 26f\n"
      "tbz x10, #3, 21f\n"
      "str d8, [x11], #0x8\n"
      "tbz x10, #2, 19f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "tbz x10, #1, 18f\n"
      "st1 { v8.h }[6], [x11], #0x2\n"
      "tbz x10, #0, 25f\n"
      "st1 { v8.b }[14], [x11]\n"
      "b 25f\n"
      "18:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x10, #0, 25f\n"
      "st1 { v8.b }[12], [x11]\n"
      "b 25f\n"
      "19:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x10, #1, 20f\n"
      "st1 { v8.h }[4], [x11], #0x2\n"
      "tbz x10, #0, 25f\n"
      "st1 { v8.b }[10], [x11]\n"
      "b 25f\n"
      "20:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x10, #0, 25f\n"
      "st1 { v8.b }[8], [x11]\n"
      "b 25f\n"
      "21:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x10, #2, 23f\n"
      "str s8, [x11], #0x4\n"
      "tbz x10, #1, 22f\n"
      "st1 { v8.h }[2], [x11], #0x2\n"
      "tbz x10, #0, 25f\n"
      "st1 { v8.b }[6], [x11]\n"
      "b 25f\n"
      "22:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x10, #0, 25f\n"
      "st1 { v8.b }[4], [x11]\n"
      "b 25f\n"
      "23:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x10, #1, 24f\n"
      "str h8, [x11], #0x2\n"
      "tbz x10, #0, 25f\n"
      "st1 { v8.b }[2], [x11]\n"
      "b 25f\n"
      "24:"  // Height 1: Partial direct writeback: partial_1_0
      "str b8, [x11, #0x0]\n"
      "25:"  // Height 1: Partial direct writeback: Done
      "b 27f\n"
      "26:"  // Height 1: Full writeback
      "str q8, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "27:"  // Height 1: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 2b\n"
      "b 164f\n"
      "28:"  // Height 2
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "29:"  // Height 2: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "30:"  // Height 2: setup done
      "mov x28, #0x0\n"
      "31:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 32f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 33f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "b 33f\n"
      "32:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "33:"  // Height 2: input setup done
      "cmp x27, #0x10\n"
      "blt 36f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q6, [x9, #0x0]\n"
      "ldr q7, [x9, #0x10]\n"
      "blt 35f\n"
      "34:"  // Height 2: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q17, [x9, #0x20]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q16, [x9, #0x30]\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f80e22a  // sdot v10.4s, v17.16b, v0.4b[0]\n"
      ".inst 0x4f81e22e  // sdot v14.4s, v17.16b, v1.4b[0]\n"
      "ldr q17, [x9, #0x40]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f80e20b  // sdot v11.4s, v16.16b, v0.4b[0]\n"
      ".inst 0x4f81e20f  // sdot v15.4s, v16.16b, v1.4b[0]\n"
      "ldr q16, [x9, #0x50]\n"
      "cmp x27, #0x20\n"
      ".inst 0x4fa0e228  // sdot v8.4s, v17.16b, v0.4b[1]\n"
      ".inst 0x4fa1e22c  // sdot v12.4s, v17.16b, v1.4b[1]\n"
      "ldr q17, [x9, #0x60]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4fa0e209  // sdot v9.4s, v16.16b, v0.4b[1]\n"
      ".inst 0x4fa1e20d  // sdot v13.4s, v16.16b, v1.4b[1]\n"
      "ldr q16, [x9, #0x70]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4fa0e22a  // sdot v10.4s, v17.16b, v0.4b[1]\n"
      ".inst 0x4fa1e22e  // sdot v14.4s, v17.16b, v1.4b[1]\n"
      "ldr q17, [x9, #0x80]\n"
      ".inst 0x4fa0e20b  // sdot v11.4s, v16.16b, v0.4b[1]\n"
      ".inst 0x4fa1e20f  // sdot v15.4s, v16.16b, v1.4b[1]\n"
      "ldr q16, [x9, #0x90]\n"
      ".inst 0x4f80ea28  // sdot v8.4s, v17.16b, v0.4b[2]\n"
      ".inst 0x4f81ea2c  // sdot v12.4s, v17.16b, v1.4b[2]\n"
      "ldr q17, [x9, #0xa0]\n"
      ".inst 0x4f80ea09  // sdot v9.4s, v16.16b, v0.4b[2]\n"
      ".inst 0x4f81ea0d  // sdot v13.4s, v16.16b, v1.4b[2]\n"
      "ldr q16, [x9, #0xb0]\n"
      ".inst 0x4f80ea2a  // sdot v10.4s, v17.16b, v0.4b[2]\n"
      ".inst 0x4f81ea2e  // sdot v14.4s, v17.16b, v1.4b[2]\n"
      "ldr q17, [x9, #0xc0]\n"
      ".inst 0x4f80ea0b  // sdot v11.4s, v16.16b, v0.4b[2]\n"
      ".inst 0x4f81ea0f  // sdot v15.4s, v16.16b, v1.4b[2]\n"
      "ldr q16, [x9, #0xd0]\n"
      ".inst 0x4fa0ea28  // sdot v8.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x4fa1ea2c  // sdot v12.4s, v17.16b, v1.4b[3]\n"
      "ldr q17, [x9, #0xe0]\n"
      ".inst 0x4fa0ea09  // sdot v9.4s, v16.16b, v0.4b[3]\n"
      ".inst 0x4fa1ea0d  // sdot v13.4s, v16.16b, v1.4b[3]\n"
      "ldr q16, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4fa0ea2a  // sdot v10.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x4fa1ea2e  // sdot v14.4s, v17.16b, v1.4b[3]\n"
      "ldr q6, [x9, #0x0]\n"
      ".inst 0x4fa0ea0b  // sdot v11.4s, v16.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4fa1ea0f  // sdot v15.4s, v16.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      "ldr q7, [x9, #0x10]\n"
      "bge 34b\n"
      "35:"  // Height 2: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q17, [x9, #0x20]\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q16, [x9, #0x30]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f80e22a  // sdot v10.4s, v17.16b, v0.4b[0]\n"
      ".inst 0x4f81e22e  // sdot v14.4s, v17.16b, v1.4b[0]\n"
      "ldr q17, [x9, #0x40]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4f80e20b  // sdot v11.4s, v16.16b, v0.4b[0]\n"
      ".inst 0x4f81e20f  // sdot v15.4s, v16.16b, v1.4b[0]\n"
      "ldr q16, [x9, #0x50]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4fa0e228  // sdot v8.4s, v17.16b, v0.4b[1]\n"
      ".inst 0x4fa1e22c  // sdot v12.4s, v17.16b, v1.4b[1]\n"
      "ldr q17, [x9, #0x60]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4fa0e209  // sdot v9.4s, v16.16b, v0.4b[1]\n"
      ".inst 0x4fa1e20d  // sdot v13.4s, v16.16b, v1.4b[1]\n"
      "ldr q16, [x9, #0x70]\n"
      ".inst 0x4fa0e22a  // sdot v10.4s, v17.16b, v0.4b[1]\n"
      ".inst 0x4fa1e22e  // sdot v14.4s, v17.16b, v1.4b[1]\n"
      "ldr q17, [x9, #0x80]\n"
      ".inst 0x4fa0e20b  // sdot v11.4s, v16.16b, v0.4b[1]\n"
      ".inst 0x4fa1e20f  // sdot v15.4s, v16.16b, v1.4b[1]\n"
      "ldr q16, [x9, #0x90]\n"
      ".inst 0x4f80ea28  // sdot v8.4s, v17.16b, v0.4b[2]\n"
      ".inst 0x4f81ea2c  // sdot v12.4s, v17.16b, v1.4b[2]\n"
      "ldr q17, [x9, #0xa0]\n"
      ".inst 0x4f80ea09  // sdot v9.4s, v16.16b, v0.4b[2]\n"
      ".inst 0x4f81ea0d  // sdot v13.4s, v16.16b, v1.4b[2]\n"
      "ldr q16, [x9, #0xb0]\n"
      ".inst 0x4f80ea2a  // sdot v10.4s, v17.16b, v0.4b[2]\n"
      ".inst 0x4f81ea2e  // sdot v14.4s, v17.16b, v1.4b[2]\n"
      "ldr q17, [x9, #0xc0]\n"
      ".inst 0x4f80ea0b  // sdot v11.4s, v16.16b, v0.4b[2]\n"
      ".inst 0x4f81ea0f  // sdot v15.4s, v16.16b, v1.4b[2]\n"
      "ldr q16, [x9, #0xd0]\n"
      ".inst 0x4fa0ea28  // sdot v8.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x4fa1ea2c  // sdot v12.4s, v17.16b, v1.4b[3]\n"
      "ldr q17, [x9, #0xe0]\n"
      ".inst 0x4fa0ea09  // sdot v9.4s, v16.16b, v0.4b[3]\n"
      ".inst 0x4fa1ea0d  // sdot v13.4s, v16.16b, v1.4b[3]\n"
      "ldr q16, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4fa0ea2a  // sdot v10.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x4fa1ea2e  // sdot v14.4s, v17.16b, v1.4b[3]\n"
      ".inst 0x4fa0ea0b  // sdot v11.4s, v16.16b, v0.4b[3]\n"
      ".inst 0x4fa1ea0f  // sdot v15.4s, v16.16b, v1.4b[3]\n"
      "36:"  // Height 2: Multiply loop: Main loop skip
      "cbz x27, 41f\n"
      "cmp x27, #0x4\n"
      "blt 38f\n"
      "37:"  // Height 2: Multiply loop: Odd block loop
      "ldr s19, [x26], #0x4\n"
      "ldr s18, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "cmp x27, #0x4\n"
      "ldr q17, [x9, #0x0]\n"
      "ldr q16, [x9, #0x10]\n"
      ".inst 0x4f93e228  // sdot v8.4s, v17.16b, v19.4b[0]\n"
      ".inst 0x4f92e22c  // sdot v12.4s, v17.16b, v18.4b[0]\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4f93e209  // sdot v9.4s, v16.16b, v19.4b[0]\n"
      ".inst 0x4f92e20d  // sdot v13.4s, v16.16b, v18.4b[0]\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4f93e22a  // sdot v10.4s, v17.16b, v19.4b[0]\n"
      ".inst 0x4f92e22e  // sdot v14.4s, v17.16b, v18.4b[0]\n"
      "add x9, x9, #0x40\n"
      ".inst 0x4f93e20b  // sdot v11.4s, v16.16b, v19.4b[0]\n"
      ".inst 0x4f92e20f  // sdot v15.4s, v16.16b, v18.4b[0]\n"
      "bge 37b\n"
      "38:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x27, 41f\n"
      "tbz x27, #1, 39f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "tbz x27, #0, 40f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "b 40f\n"
      "39:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "40:"  // Height 2: Multiply loop: Ragged operand read: Done
      "ldr q17, [x9, #0x0]\n"
      "ldr q16, [x9, #0x10]\n"
      ".inst 0x4f80e228  // sdot v8.4s, v17.16b, v0.4b[0]\n"
      ".inst 0x4f81e22c  // sdot v12.4s, v17.16b, v1.4b[0]\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4f80e209  // sdot v9.4s, v16.16b, v0.4b[0]\n"
      ".inst 0x4f81e20d  // sdot v13.4s, v16.16b, v1.4b[0]\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4f80e22a  // sdot v10.4s, v17.16b, v0.4b[0]\n"
      ".inst 0x4f81e22e  // sdot v14.4s, v17.16b, v1.4b[0]\n"
      "add x9, x9, #0x40\n"
      ".inst 0x4f80e20b  // sdot v11.4s, v16.16b, v0.4b[0]\n"
      ".inst 0x4f81e20f  // sdot v15.4s, v16.16b, v1.4b[0]\n"
      "41:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 31b\n"
      "ldr q19, [x14, #0x0]\n"
      "ldr q18, [x14, #0x10]\n"
      "add v8.4s, v8.4s, v19.4s\n"
      "add v9.4s, v9.4s, v18.4s\n"
      "ldr q17, [x14, #0x20]\n"
      "ldr q16, [x14, #0x30]\n"
      "add v10.4s, v10.4s, v17.4s\n"
      "add v11.4s, v11.4s, v16.4s\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x11, x20\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "add v12.4s, v12.4s, v19.4s\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "add v13.4s, v13.4s, v18.4s\n"
      "add v14.4s, v14.4s, v17.4s\n"
      "add x14, x14, #0x40\n"
      "add v15.4s, v15.4s, v16.4s\n"
      "tbz %x[flags], #4, 42f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 43f\n"
      "42:"  // Height 2: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "43:"  // Height 2: parameters loaded
      "sqrdmulh v8.4s, v8.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "sqrdmulh v12.4s, v12.4s, v4.4s\n"
      "sqrdmulh v13.4s, v13.4s, v5.4s\n"
      "sqrdmulh v14.4s, v14.4s, v6.4s\n"
      "sqrdmulh v15.4s, v15.4s, v7.4s\n"
      "tbz %x[flags], #5, 44f\n"
      "and v19.16b, v8.16b, v0.16b\n"
      "and v18.16b, v9.16b, v1.16b\n"
      "and v17.16b, v10.16b, v2.16b\n"
      "and v16.16b, v11.16b, v3.16b\n"
      "sshr v19.4s, v19.4s, #0x1f\n"
      "sshr v18.4s, v18.4s, #0x1f\n"
      "sshr v17.4s, v17.4s, #0x1f\n"
      "sshr v16.4s, v16.4s, #0x1f\n"
      "sqadd v8.4s, v8.4s, v19.4s\n"
      "sqadd v9.4s, v9.4s, v18.4s\n"
      "sqadd v10.4s, v10.4s, v17.4s\n"
      "sqadd v11.4s, v11.4s, v16.4s\n"
      "and v19.16b, v12.16b, v0.16b\n"
      "and v18.16b, v13.16b, v1.16b\n"
      "and v17.16b, v14.16b, v2.16b\n"
      "and v16.16b, v15.16b, v3.16b\n"
      "sshr v19.4s, v19.4s, #0x1f\n"
      "sshr v18.4s, v18.4s, #0x1f\n"
      "sshr v17.4s, v17.4s, #0x1f\n"
      "sshr v16.4s, v16.4s, #0x1f\n"
      "sqadd v12.4s, v12.4s, v19.4s\n"
      "sqadd v13.4s, v13.4s, v18.4s\n"
      "sqadd v14.4s, v14.4s, v17.4s\n"
      "sqadd v15.4s, v15.4s, v16.4s\n"
      "44:"  // Height 2: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v18.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v17.4s }, [x20]\n"
      "srshl v12.4s, v12.4s, v0.4s\n"
      "srshl v13.4s, v13.4s, v1.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v16.4s }, [x20]\n"
      "srshl v14.4s, v14.4s, v2.4s\n"
      "srshl v15.4s, v15.4s, v3.4s\n"
      "cmp x10, #0x10\n"
      "add v8.4s, v8.4s, v18.4s\n"
      "add v9.4s, v9.4s, v18.4s\n"
      "add v10.4s, v10.4s, v18.4s\n"
      "add v11.4s, v11.4s, v18.4s\n"
      "add v12.4s, v12.4s, v18.4s\n"
      "add v13.4s, v13.4s, v18.4s\n"
      "add v14.4s, v14.4s, v18.4s\n"
      "add v15.4s, v15.4s, v18.4s\n"
      "smin v8.4s, v8.4s, v17.4s\n"
      "smin v9.4s, v9.4s, v17.4s\n"
      "smin v10.4s, v10.4s, v17.4s\n"
      "smin v11.4s, v11.4s, v17.4s\n"
      "smin v12.4s, v12.4s, v17.4s\n"
      "smin v13.4s, v13.4s, v17.4s\n"
      "smin v14.4s, v14.4s, v17.4s\n"
      "smin v15.4s, v15.4s, v17.4s\n"
      "smax v8.4s, v8.4s, v16.4s\n"
      "smax v9.4s, v9.4s, v16.4s\n"
      "smax v10.4s, v10.4s, v16.4s\n"
      "smax v11.4s, v11.4s, v16.4s\n"
      "smax v12.4s, v12.4s, v16.4s\n"
      "smax v13.4s, v13.4s, v16.4s\n"
      "smax v14.4s, v14.4s, v16.4s\n"
      "smax v15.4s, v15.4s, v16.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v17.8h, v10.8h, v11.8h\n"
      "uzp1 v12.8h, v12.8h, v13.8h\n"
      "uzp1 v16.8h, v14.8h, v15.8h\n"
      "uzp1 v8.16b, v8.16b, v17.16b\n"
      "uzp1 v12.16b, v12.16b, v16.16b\n"
      "bge 53f\n"
      "tbz x10, #3, 48f\n"
      "str d8, [x11], #0x8\n"
      "str d12, [x25], #0x8\n"
      "tbz x10, #2, 46f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "st1 { v12.s }[2], [x25], #0x4\n"
      "tbz x10, #1, 45f\n"
      "st1 { v8.h }[6], [x11], #0x2\n"
      "st1 { v12.h }[6], [x25], #0x2\n"
      "tbz x10, #0, 52f\n"
      "st1 { v8.b }[14], [x11]\n"
      "st1 { v12.b }[14], [x25]\n"
      "b 52f\n"
      "45:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x10, #0, 52f\n"
      "st1 { v8.b }[12], [x11]\n"
      "st1 { v12.b }[12], [x25]\n"
      "b 52f\n"
      "46:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x10, #1, 47f\n"
      "st1 { v8.h }[4], [x11], #0x2\n"
      "st1 { v12.h }[4], [x25], #0x2\n"
      "tbz x10, #0, 52f\n"
      "st1 { v8.b }[10], [x11]\n"
      "st1 { v12.b }[10], [x25]\n"
      "b 52f\n"
      "47:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x10, #0, 52f\n"
      "st1 { v8.b }[8], [x11]\n"
      "st1 { v12.b }[8], [x25]\n"
      "b 52f\n"
      "48:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x10, #2, 50f\n"
      "str s8, [x11], #0x4\n"
      "str s12, [x25], #0x4\n"
      "tbz x10, #1, 49f\n"
      "st1 { v8.h }[2], [x11], #0x2\n"
      "st1 { v12.h }[2], [x25], #0x2\n"
      "tbz x10, #0, 52f\n"
      "st1 { v8.b }[6], [x11]\n"
      "st1 { v12.b }[6], [x25]\n"
      "b 52f\n"
      "49:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x10, #0, 52f\n"
      "st1 { v8.b }[4], [x11]\n"
      "st1 { v12.b }[4], [x25]\n"
      "b 52f\n"
      "50:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x10, #1, 51f\n"
      "str h8, [x11], #0x2\n"
      "str h12, [x25], #0x2\n"
      "tbz x10, #0, 52f\n"
      "st1 { v8.b }[2], [x11]\n"
      "st1 { v12.b }[2], [x25]\n"
      "b 52f\n"
      "51:"  // Height 2: Partial direct writeback: partial_1_0
      "str b8, [x11, #0x0]\n"
      "str b12, [x25, #0x0]\n"
      "52:"  // Height 2: Partial direct writeback: Done
      "b 54f\n"
      "53:"  // Height 2: Full writeback
      "str q8, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q12, [x25, #0x0]\n"
      "54:"  // Height 2: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 29b\n"
      "b 164f\n"
      "55:"  // Height 3
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "56:"  // Height 3: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "57:"  // Height 3: setup done
      "mov x28, #0x0\n"
      "58:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 59f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 60f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "b 60f\n"
      "59:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "60:"  // Height 3: input setup done
      "cmp x27, #0x10\n"
      "blt 63f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q6, [x9, #0x0]\n"
      "ldr q7, [x9, #0x10]\n"
      "blt 62f\n"
      "61:"  // Height 3: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q21, [x9, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q20, [x9, #0x30]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f80e2aa  // sdot v10.4s, v21.16b, v0.4b[0]\n"
      ".inst 0x4f81e2ae  // sdot v14.4s, v21.16b, v1.4b[0]\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f82e2b2  // sdot v18.4s, v21.16b, v2.4b[0]\n"
      "ldr q21, [x9, #0x40]\n"
      ".inst 0x4f80e28b  // sdot v11.4s, v20.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f81e28f  // sdot v15.4s, v20.16b, v1.4b[0]\n"
      ".inst 0x4f82e293  // sdot v19.4s, v20.16b, v2.4b[0]\n"
      "ldr q20, [x9, #0x50]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4fa0e2a8  // sdot v8.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x4fa1e2ac  // sdot v12.4s, v21.16b, v1.4b[1]\n"
      ".inst 0x4fa2e2b0  // sdot v16.4s, v21.16b, v2.4b[1]\n"
      "ldr q21, [x9, #0x60]\n"
      ".inst 0x4fa0e289  // sdot v9.4s, v20.16b, v0.4b[1]\n"
      ".inst 0x4fa1e28d  // sdot v13.4s, v20.16b, v1.4b[1]\n"
      ".inst 0x4fa2e291  // sdot v17.4s, v20.16b, v2.4b[1]\n"
      "ldr q20, [x9, #0x70]\n"
      ".inst 0x4fa0e2aa  // sdot v10.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x4fa1e2ae  // sdot v14.4s, v21.16b, v1.4b[1]\n"
      ".inst 0x4fa2e2b2  // sdot v18.4s, v21.16b, v2.4b[1]\n"
      "ldr q21, [x9, #0x80]\n"
      ".inst 0x4fa0e28b  // sdot v11.4s, v20.16b, v0.4b[1]\n"
      ".inst 0x4fa1e28f  // sdot v15.4s, v20.16b, v1.4b[1]\n"
      ".inst 0x4fa2e293  // sdot v19.4s, v20.16b, v2.4b[1]\n"
      "ldr q20, [x9, #0x90]\n"
      ".inst 0x4f80eaa8  // sdot v8.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x4f81eaac  // sdot v12.4s, v21.16b, v1.4b[2]\n"
      ".inst 0x4f82eab0  // sdot v16.4s, v21.16b, v2.4b[2]\n"
      "ldr q21, [x9, #0xa0]\n"
      ".inst 0x4f80ea89  // sdot v9.4s, v20.16b, v0.4b[2]\n"
      ".inst 0x4f81ea8d  // sdot v13.4s, v20.16b, v1.4b[2]\n"
      ".inst 0x4f82ea91  // sdot v17.4s, v20.16b, v2.4b[2]\n"
      "ldr q20, [x9, #0xb0]\n"
      ".inst 0x4f80eaaa  // sdot v10.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x4f81eaae  // sdot v14.4s, v21.16b, v1.4b[2]\n"
      ".inst 0x4f82eab2  // sdot v18.4s, v21.16b, v2.4b[2]\n"
      "ldr q21, [x9, #0xc0]\n"
      ".inst 0x4f80ea8b  // sdot v11.4s, v20.16b, v0.4b[2]\n"
      ".inst 0x4f81ea8f  // sdot v15.4s, v20.16b, v1.4b[2]\n"
      ".inst 0x4f82ea93  // sdot v19.4s, v20.16b, v2.4b[2]\n"
      "ldr q20, [x9, #0xd0]\n"
      ".inst 0x4fa0eaa8  // sdot v8.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x4fa1eaac  // sdot v12.4s, v21.16b, v1.4b[3]\n"
      ".inst 0x4fa2eab0  // sdot v16.4s, v21.16b, v2.4b[3]\n"
      "ldr q21, [x9, #0xe0]\n"
      ".inst 0x4fa0ea89  // sdot v9.4s, v20.16b, v0.4b[3]\n"
      ".inst 0x4fa1ea8d  // sdot v13.4s, v20.16b, v1.4b[3]\n"
      ".inst 0x4fa2ea91  // sdot v17.4s, v20.16b, v2.4b[3]\n"
      "ldr q20, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4fa0eaaa  // sdot v10.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x4fa1eaae  // sdot v14.4s, v21.16b, v1.4b[3]\n"
      ".inst 0x4fa2eab2  // sdot v18.4s, v21.16b, v2.4b[3]\n"
      "ldr q6, [x9, #0x0]\n"
      ".inst 0x4fa0ea8b  // sdot v11.4s, v20.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4fa1ea8f  // sdot v15.4s, v20.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4fa2ea93  // sdot v19.4s, v20.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q7, [x9, #0x10]\n"
      "bge 61b\n"
      "62:"  // Height 3: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q21, [x9, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q20, [x9, #0x30]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4f80e2aa  // sdot v10.4s, v21.16b, v0.4b[0]\n"
      ".inst 0x4f81e2ae  // sdot v14.4s, v21.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f82e2b2  // sdot v18.4s, v21.16b, v2.4b[0]\n"
      "ldr q21, [x9, #0x40]\n"
      ".inst 0x4f80e28b  // sdot v11.4s, v20.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f81e28f  // sdot v15.4s, v20.16b, v1.4b[0]\n"
      ".inst 0x4f82e293  // sdot v19.4s, v20.16b, v2.4b[0]\n"
      "ldr q20, [x9, #0x50]\n"
      ".inst 0x4fa0e2a8  // sdot v8.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x4fa1e2ac  // sdot v12.4s, v21.16b, v1.4b[1]\n"
      ".inst 0x4fa2e2b0  // sdot v16.4s, v21.16b, v2.4b[1]\n"
      "ldr q21, [x9, #0x60]\n"
      ".inst 0x4fa0e289  // sdot v9.4s, v20.16b, v0.4b[1]\n"
      ".inst 0x4fa1e28d  // sdot v13.4s, v20.16b, v1.4b[1]\n"
      ".inst 0x4fa2e291  // sdot v17.4s, v20.16b, v2.4b[1]\n"
      "ldr q20, [x9, #0x70]\n"
      ".inst 0x4fa0e2aa  // sdot v10.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x4fa1e2ae  // sdot v14.4s, v21.16b, v1.4b[1]\n"
      ".inst 0x4fa2e2b2  // sdot v18.4s, v21.16b, v2.4b[1]\n"
      "ldr q21, [x9, #0x80]\n"
      ".inst 0x4fa0e28b  // sdot v11.4s, v20.16b, v0.4b[1]\n"
      ".inst 0x4fa1e28f  // sdot v15.4s, v20.16b, v1.4b[1]\n"
      ".inst 0x4fa2e293  // sdot v19.4s, v20.16b, v2.4b[1]\n"
      "ldr q20, [x9, #0x90]\n"
      ".inst 0x4f80eaa8  // sdot v8.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x4f81eaac  // sdot v12.4s, v21.16b, v1.4b[2]\n"
      ".inst 0x4f82eab0  // sdot v16.4s, v21.16b, v2.4b[2]\n"
      "ldr q21, [x9, #0xa0]\n"
      ".inst 0x4f80ea89  // sdot v9.4s, v20.16b, v0.4b[2]\n"
      ".inst 0x4f81ea8d  // sdot v13.4s, v20.16b, v1.4b[2]\n"
      ".inst 0x4f82ea91  // sdot v17.4s, v20.16b, v2.4b[2]\n"
      "ldr q20, [x9, #0xb0]\n"
      ".inst 0x4f80eaaa  // sdot v10.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x4f81eaae  // sdot v14.4s, v21.16b, v1.4b[2]\n"
      ".inst 0x4f82eab2  // sdot v18.4s, v21.16b, v2.4b[2]\n"
      "ldr q21, [x9, #0xc0]\n"
      ".inst 0x4f80ea8b  // sdot v11.4s, v20.16b, v0.4b[2]\n"
      ".inst 0x4f81ea8f  // sdot v15.4s, v20.16b, v1.4b[2]\n"
      ".inst 0x4f82ea93  // sdot v19.4s, v20.16b, v2.4b[2]\n"
      "ldr q20, [x9, #0xd0]\n"
      ".inst 0x4fa0eaa8  // sdot v8.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x4fa1eaac  // sdot v12.4s, v21.16b, v1.4b[3]\n"
      ".inst 0x4fa2eab0  // sdot v16.4s, v21.16b, v2.4b[3]\n"
      "ldr q21, [x9, #0xe0]\n"
      ".inst 0x4fa0ea89  // sdot v9.4s, v20.16b, v0.4b[3]\n"
      ".inst 0x4fa1ea8d  // sdot v13.4s, v20.16b, v1.4b[3]\n"
      ".inst 0x4fa2ea91  // sdot v17.4s, v20.16b, v2.4b[3]\n"
      "ldr q20, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4fa0eaaa  // sdot v10.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x4fa1eaae  // sdot v14.4s, v21.16b, v1.4b[3]\n"
      ".inst 0x4fa2eab2  // sdot v18.4s, v21.16b, v2.4b[3]\n"
      ".inst 0x4fa0ea8b  // sdot v11.4s, v20.16b, v0.4b[3]\n"
      ".inst 0x4fa1ea8f  // sdot v15.4s, v20.16b, v1.4b[3]\n"
      ".inst 0x4fa2ea93  // sdot v19.4s, v20.16b, v2.4b[3]\n"
      "63:"  // Height 3: Multiply loop: Main loop skip
      "cbz x27, 68f\n"
      "cmp x27, #0x4\n"
      "blt 65f\n"
      "64:"  // Height 3: Multiply loop: Odd block loop
      "ldr s24, [x26], #0x4\n"
      "ldr s23, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "cmp x27, #0x4\n"
      "ldr s22, [x24], #0x4\n"
      "ldr q21, [x9, #0x0]\n"
      ".inst 0x4f98e2a8  // sdot v8.4s, v21.16b, v24.4b[0]\n"
      ".inst 0x4f97e2ac  // sdot v12.4s, v21.16b, v23.4b[0]\n"
      "ldr q20, [x9, #0x10]\n"
      ".inst 0x4f96e2b0  // sdot v16.4s, v21.16b, v22.4b[0]\n"
      "ldr q21, [x9, #0x20]\n"
      ".inst 0x4f98e289  // sdot v9.4s, v20.16b, v24.4b[0]\n"
      ".inst 0x4f97e28d  // sdot v13.4s, v20.16b, v23.4b[0]\n"
      ".inst 0x4f96e291  // sdot v17.4s, v20.16b, v22.4b[0]\n"
      "ldr q20, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      ".inst 0x4f98e2aa  // sdot v10.4s, v21.16b, v24.4b[0]\n"
      ".inst 0x4f97e2ae  // sdot v14.4s, v21.16b, v23.4b[0]\n"
      ".inst 0x4f96e2b2  // sdot v18.4s, v21.16b, v22.4b[0]\n"
      ".inst 0x4f98e28b  // sdot v11.4s, v20.16b, v24.4b[0]\n"
      ".inst 0x4f97e28f  // sdot v15.4s, v20.16b, v23.4b[0]\n"
      ".inst 0x4f96e293  // sdot v19.4s, v20.16b, v22.4b[0]\n"
      "bge 64b\n"
      "65:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x27, 68f\n"
      "tbz x27, #1, 66f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "tbz x27, #0, 67f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "b 67f\n"
      "66:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "67:"  // Height 3: Multiply loop: Ragged operand read: Done
      "ldr q21, [x9, #0x0]\n"
      "ldr q20, [x9, #0x10]\n"
      ".inst 0x4f80e2a8  // sdot v8.4s, v21.16b, v0.4b[0]\n"
      ".inst 0x4f81e2ac  // sdot v12.4s, v21.16b, v1.4b[0]\n"
      ".inst 0x4f82e2b0  // sdot v16.4s, v21.16b, v2.4b[0]\n"
      "ldr q21, [x9, #0x20]\n"
      ".inst 0x4f80e289  // sdot v9.4s, v20.16b, v0.4b[0]\n"
      ".inst 0x4f81e28d  // sdot v13.4s, v20.16b, v1.4b[0]\n"
      ".inst 0x4f82e291  // sdot v17.4s, v20.16b, v2.4b[0]\n"
      "ldr q20, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      ".inst 0x4f80e2aa  // sdot v10.4s, v21.16b, v0.4b[0]\n"
      ".inst 0x4f81e2ae  // sdot v14.4s, v21.16b, v1.4b[0]\n"
      ".inst 0x4f82e2b2  // sdot v18.4s, v21.16b, v2.4b[0]\n"
      ".inst 0x4f80e28b  // sdot v11.4s, v20.16b, v0.4b[0]\n"
      ".inst 0x4f81e28f  // sdot v15.4s, v20.16b, v1.4b[0]\n"
      ".inst 0x4f82e293  // sdot v19.4s, v20.16b, v2.4b[0]\n"
      "68:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 58b\n"
      "ldr q23, [x14, #0x0]\n"
      "ldr q22, [x14, #0x10]\n"
      "add v8.4s, v8.4s, v23.4s\n"
      "add v9.4s, v9.4s, v22.4s\n"
      "ldr q21, [x14, #0x20]\n"
      "ldr q20, [x14, #0x30]\n"
      "add v10.4s, v10.4s, v21.4s\n"
      "add v11.4s, v11.4s, v20.4s\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x11, x20\n"
      "add x24, x25, x20\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add v12.4s, v12.4s, v23.4s\n"
      "add v13.4s, v13.4s, v22.4s\n"
      "add v14.4s, v14.4s, v21.4s\n"
      "add v15.4s, v15.4s, v20.4s\n"
      "add x14, x14, #0x40\n"
      "add v16.4s, v16.4s, v23.4s\n"
      "add v17.4s, v17.4s, v22.4s\n"
      "add v18.4s, v18.4s, v21.4s\n"
      "add v19.4s, v19.4s, v20.4s\n"
      "tbz %x[flags], #4, 69f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 70f\n"
      "69:"  // Height 3: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "70:"  // Height 3: parameters loaded
      "sqrdmulh v8.4s, v8.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "sqrdmulh v12.4s, v12.4s, v4.4s\n"
      "sqrdmulh v13.4s, v13.4s, v5.4s\n"
      "sqrdmulh v14.4s, v14.4s, v6.4s\n"
      "sqrdmulh v15.4s, v15.4s, v7.4s\n"
      "sqrdmulh v16.4s, v16.4s, v4.4s\n"
      "sqrdmulh v17.4s, v17.4s, v5.4s\n"
      "sqrdmulh v18.4s, v18.4s, v6.4s\n"
      "sqrdmulh v19.4s, v19.4s, v7.4s\n"
      "tbz %x[flags], #5, 71f\n"
      "and v23.16b, v8.16b, v0.16b\n"
      "and v22.16b, v9.16b, v1.16b\n"
      "and v21.16b, v10.16b, v2.16b\n"
      "and v20.16b, v11.16b, v3.16b\n"
      "sshr v23.4s, v23.4s, #0x1f\n"
      "sshr v22.4s, v22.4s, #0x1f\n"
      "sshr v21.4s, v21.4s, #0x1f\n"
      "sshr v20.4s, v20.4s, #0x1f\n"
      "sqadd v8.4s, v8.4s, v23.4s\n"
      "sqadd v9.4s, v9.4s, v22.4s\n"
      "sqadd v10.4s, v10.4s, v21.4s\n"
      "sqadd v11.4s, v11.4s, v20.4s\n"
      "and v23.16b, v12.16b, v0.16b\n"
      "and v22.16b, v13.16b, v1.16b\n"
      "and v21.16b, v14.16b, v2.16b\n"
      "and v20.16b, v15.16b, v3.16b\n"
      "sshr v23.4s, v23.4s, #0x1f\n"
      "sshr v22.4s, v22.4s, #0x1f\n"
      "sshr v21.4s, v21.4s, #0x1f\n"
      "sshr v20.4s, v20.4s, #0x1f\n"
      "sqadd v12.4s, v12.4s, v23.4s\n"
      "sqadd v13.4s, v13.4s, v22.4s\n"
      "sqadd v14.4s, v14.4s, v21.4s\n"
      "sqadd v15.4s, v15.4s, v20.4s\n"
      "and v23.16b, v16.16b, v0.16b\n"
      "and v22.16b, v17.16b, v1.16b\n"
      "and v21.16b, v18.16b, v2.16b\n"
      "and v20.16b, v19.16b, v3.16b\n"
      "sshr v23.4s, v23.4s, #0x1f\n"
      "sshr v22.4s, v22.4s, #0x1f\n"
      "sshr v21.4s, v21.4s, #0x1f\n"
      "sshr v20.4s, v20.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v23.4s\n"
      "sqadd v17.4s, v17.4s, v22.4s\n"
      "sqadd v18.4s, v18.4s, v21.4s\n"
      "sqadd v19.4s, v19.4s, v20.4s\n"
      "71:"  // Height 3: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v22.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v21.4s }, [x20]\n"
      "srshl v12.4s, v12.4s, v0.4s\n"
      "srshl v13.4s, v13.4s, v1.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v20.4s }, [x20]\n"
      "srshl v14.4s, v14.4s, v2.4s\n"
      "srshl v15.4s, v15.4s, v3.4s\n"
      "cmp x10, #0x10\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "add v8.4s, v8.4s, v22.4s\n"
      "add v9.4s, v9.4s, v22.4s\n"
      "add v10.4s, v10.4s, v22.4s\n"
      "add v11.4s, v11.4s, v22.4s\n"
      "add v12.4s, v12.4s, v22.4s\n"
      "add v13.4s, v13.4s, v22.4s\n"
      "add v14.4s, v14.4s, v22.4s\n"
      "add v15.4s, v15.4s, v22.4s\n"
      "add v16.4s, v16.4s, v22.4s\n"
      "add v17.4s, v17.4s, v22.4s\n"
      "add v18.4s, v18.4s, v22.4s\n"
      "add v19.4s, v19.4s, v22.4s\n"
      "smin v8.4s, v8.4s, v21.4s\n"
      "smin v9.4s, v9.4s, v21.4s\n"
      "smin v10.4s, v10.4s, v21.4s\n"
      "smin v11.4s, v11.4s, v21.4s\n"
      "smin v12.4s, v12.4s, v21.4s\n"
      "smin v13.4s, v13.4s, v21.4s\n"
      "smin v14.4s, v14.4s, v21.4s\n"
      "smin v15.4s, v15.4s, v21.4s\n"
      "smin v16.4s, v16.4s, v21.4s\n"
      "smin v17.4s, v17.4s, v21.4s\n"
      "smin v18.4s, v18.4s, v21.4s\n"
      "smin v19.4s, v19.4s, v21.4s\n"
      "smax v8.4s, v8.4s, v20.4s\n"
      "smax v9.4s, v9.4s, v20.4s\n"
      "smax v10.4s, v10.4s, v20.4s\n"
      "smax v11.4s, v11.4s, v20.4s\n"
      "smax v12.4s, v12.4s, v20.4s\n"
      "smax v13.4s, v13.4s, v20.4s\n"
      "smax v14.4s, v14.4s, v20.4s\n"
      "smax v15.4s, v15.4s, v20.4s\n"
      "smax v16.4s, v16.4s, v20.4s\n"
      "smax v17.4s, v17.4s, v20.4s\n"
      "smax v18.4s, v18.4s, v20.4s\n"
      "smax v19.4s, v19.4s, v20.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v21.8h, v10.8h, v11.8h\n"
      "uzp1 v12.8h, v12.8h, v13.8h\n"
      "uzp1 v20.8h, v14.8h, v15.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v8.16b, v8.16b, v21.16b\n"
      "uzp1 v12.16b, v12.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "bge 80f\n"
      "tbz x10, #3, 75f\n"
      "str d8, [x11], #0x8\n"
      "str d12, [x25], #0x8\n"
      "str d16, [x24], #0x8\n"
      "tbz x10, #2, 73f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "st1 { v12.s }[2], [x25], #0x4\n"
      "st1 { v16.s }[2], [x24], #0x4\n"
      "tbz x10, #1, 72f\n"
      "st1 { v8.h }[6], [x11], #0x2\n"
      "st1 { v12.h }[6], [x25], #0x2\n"
      "st1 { v16.h }[6], [x24], #0x2\n"
      "tbz x10, #0, 79f\n"
      "st1 { v8.b }[14], [x11]\n"
      "st1 { v12.b }[14], [x25]\n"
      "st1 { v16.b }[14], [x24]\n"
      "b 79f\n"
      "72:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x10, #0, 79f\n"
      "st1 { v8.b }[12], [x11]\n"
      "st1 { v12.b }[12], [x25]\n"
      "st1 { v16.b }[12], [x24]\n"
      "b 79f\n"
      "73:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x10, #1, 74f\n"
      "st1 { v8.h }[4], [x11], #0x2\n"
      "st1 { v12.h }[4], [x25], #0x2\n"
      "st1 { v16.h }[4], [x24], #0x2\n"
      "tbz x10, #0, 79f\n"
      "st1 { v8.b }[10], [x11]\n"
      "st1 { v12.b }[10], [x25]\n"
      "st1 { v16.b }[10], [x24]\n"
      "b 79f\n"
      "74:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x10, #0, 79f\n"
      "st1 { v8.b }[8], [x11]\n"
      "st1 { v12.b }[8], [x25]\n"
      "st1 { v16.b }[8], [x24]\n"
      "b 79f\n"
      "75:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x10, #2, 77f\n"
      "str s8, [x11], #0x4\n"
      "str s12, [x25], #0x4\n"
      "str s16, [x24], #0x4\n"
      "tbz x10, #1, 76f\n"
      "st1 { v8.h }[2], [x11], #0x2\n"
      "st1 { v12.h }[2], [x25], #0x2\n"
      "st1 { v16.h }[2], [x24], #0x2\n"
      "tbz x10, #0, 79f\n"
      "st1 { v8.b }[6], [x11]\n"
      "st1 { v12.b }[6], [x25]\n"
      "st1 { v16.b }[6], [x24]\n"
      "b 79f\n"
      "76:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x10, #0, 79f\n"
      "st1 { v8.b }[4], [x11]\n"
      "st1 { v12.b }[4], [x25]\n"
      "st1 { v16.b }[4], [x24]\n"
      "b 79f\n"
      "77:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x10, #1, 78f\n"
      "str h8, [x11], #0x2\n"
      "str h12, [x25], #0x2\n"
      "str h16, [x24], #0x2\n"
      "tbz x10, #0, 79f\n"
      "st1 { v8.b }[2], [x11]\n"
      "st1 { v12.b }[2], [x25]\n"
      "st1 { v16.b }[2], [x24]\n"
      "b 79f\n"
      "78:"  // Height 3: Partial direct writeback: partial_1_0
      "str b8, [x11, #0x0]\n"
      "str b12, [x25, #0x0]\n"
      "str b16, [x24, #0x0]\n"
      "79:"  // Height 3: Partial direct writeback: Done
      "b 81f\n"
      "80:"  // Height 3: Full writeback
      "str q8, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q12, [x25, #0x0]\n"
      "str q16, [x24, #0x0]\n"
      "81:"  // Height 3: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 56b\n"
      "b 164f\n"
      "82:"  // Height 4
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "83:"  // Height 4: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "84:"  // Height 4: setup done
      "mov x28, #0x0\n"
      "85:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 86f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 87f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 87f\n"
      "86:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "87:"  // Height 4: input setup done
      "cmp x27, #0x10\n"
      "blt 90f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q6, [x9, #0x0]\n"
      "ldr q7, [x9, #0x10]\n"
      "blt 89f\n"
      "88:"  // Height 4: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q25, [x9, #0x20]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q24, [x9, #0x30]\n"
      "cmp x27, #0x20\n"
      ".inst 0x4f80e32a  // sdot v10.4s, v25.16b, v0.4b[0]\n"
      ".inst 0x4f81e32e  // sdot v14.4s, v25.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f82e332  // sdot v18.4s, v25.16b, v2.4b[0]\n"
      ".inst 0x4f83e336  // sdot v22.4s, v25.16b, v3.4b[0]\n"
      "ldr q25, [x9, #0x40]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f80e30b  // sdot v11.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x4f81e30f  // sdot v15.4s, v24.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f82e313  // sdot v19.4s, v24.16b, v2.4b[0]\n"
      ".inst 0x4f83e317  // sdot v23.4s, v24.16b, v3.4b[0]\n"
      "ldr q24, [x9, #0x50]\n"
      ".inst 0x4fa0e328  // sdot v8.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x4fa1e32c  // sdot v12.4s, v25.16b, v1.4b[1]\n"
      ".inst 0x4fa2e330  // sdot v16.4s, v25.16b, v2.4b[1]\n"
      ".inst 0x4fa3e334  // sdot v20.4s, v25.16b, v3.4b[1]\n"
      "ldr q25, [x9, #0x60]\n"
      ".inst 0x4fa0e309  // sdot v9.4s, v24.16b, v0.4b[1]\n"
      ".inst 0x4fa1e30d  // sdot v13.4s, v24.16b, v1.4b[1]\n"
      ".inst 0x4fa2e311  // sdot v17.4s, v24.16b, v2.4b[1]\n"
      ".inst 0x4fa3e315  // sdot v21.4s, v24.16b, v3.4b[1]\n"
      "ldr q24, [x9, #0x70]\n"
      ".inst 0x4fa0e32a  // sdot v10.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x4fa1e32e  // sdot v14.4s, v25.16b, v1.4b[1]\n"
      ".inst 0x4fa2e332  // sdot v18.4s, v25.16b, v2.4b[1]\n"
      ".inst 0x4fa3e336  // sdot v22.4s, v25.16b, v3.4b[1]\n"
      "ldr q25, [x9, #0x80]\n"
      ".inst 0x4fa0e30b  // sdot v11.4s, v24.16b, v0.4b[1]\n"
      ".inst 0x4fa1e30f  // sdot v15.4s, v24.16b, v1.4b[1]\n"
      ".inst 0x4fa2e313  // sdot v19.4s, v24.16b, v2.4b[1]\n"
      ".inst 0x4fa3e317  // sdot v23.4s, v24.16b, v3.4b[1]\n"
      "ldr q24, [x9, #0x90]\n"
      ".inst 0x4f80eb28  // sdot v8.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x4f81eb2c  // sdot v12.4s, v25.16b, v1.4b[2]\n"
      ".inst 0x4f82eb30  // sdot v16.4s, v25.16b, v2.4b[2]\n"
      ".inst 0x4f83eb34  // sdot v20.4s, v25.16b, v3.4b[2]\n"
      "ldr q25, [x9, #0xa0]\n"
      ".inst 0x4f80eb09  // sdot v9.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4f81eb0d  // sdot v13.4s, v24.16b, v1.4b[2]\n"
      ".inst 0x4f82eb11  // sdot v17.4s, v24.16b, v2.4b[2]\n"
      ".inst 0x4f83eb15  // sdot v21.4s, v24.16b, v3.4b[2]\n"
      "ldr q24, [x9, #0xb0]\n"
      ".inst 0x4f80eb2a  // sdot v10.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x4f81eb2e  // sdot v14.4s, v25.16b, v1.4b[2]\n"
      ".inst 0x4f82eb32  // sdot v18.4s, v25.16b, v2.4b[2]\n"
      ".inst 0x4f83eb36  // sdot v22.4s, v25.16b, v3.4b[2]\n"
      "ldr q25, [x9, #0xc0]\n"
      ".inst 0x4f80eb0b  // sdot v11.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4f81eb0f  // sdot v15.4s, v24.16b, v1.4b[2]\n"
      ".inst 0x4f82eb13  // sdot v19.4s, v24.16b, v2.4b[2]\n"
      ".inst 0x4f83eb17  // sdot v23.4s, v24.16b, v3.4b[2]\n"
      "ldr q24, [x9, #0xd0]\n"
      ".inst 0x4fa0eb28  // sdot v8.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb2c  // sdot v12.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb30  // sdot v16.4s, v25.16b, v2.4b[3]\n"
      ".inst 0x4fa3eb34  // sdot v20.4s, v25.16b, v3.4b[3]\n"
      "ldr q25, [x9, #0xe0]\n"
      ".inst 0x4fa0eb09  // sdot v9.4s, v24.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb0d  // sdot v13.4s, v24.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb11  // sdot v17.4s, v24.16b, v2.4b[3]\n"
      ".inst 0x4fa3eb15  // sdot v21.4s, v24.16b, v3.4b[3]\n"
      "ldr q24, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4fa0eb2a  // sdot v10.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb2e  // sdot v14.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb32  // sdot v18.4s, v25.16b, v2.4b[3]\n"
      ".inst 0x4fa3eb36  // sdot v22.4s, v25.16b, v3.4b[3]\n"
      "ldr q6, [x9, #0x0]\n"
      ".inst 0x4fa0eb0b  // sdot v11.4s, v24.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4fa1eb0f  // sdot v15.4s, v24.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4fa2eb13  // sdot v19.4s, v24.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      ".inst 0x4fa3eb17  // sdot v23.4s, v24.16b, v3.4b[3]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q7, [x9, #0x10]\n"
      "bge 88b\n"
      "89:"  // Height 4: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q25, [x9, #0x20]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x23, x23, #0x10\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q24, [x9, #0x30]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f80e32a  // sdot v10.4s, v25.16b, v0.4b[0]\n"
      ".inst 0x4f81e32e  // sdot v14.4s, v25.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f82e332  // sdot v18.4s, v25.16b, v2.4b[0]\n"
      ".inst 0x4f83e336  // sdot v22.4s, v25.16b, v3.4b[0]\n"
      "ldr q25, [x9, #0x40]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f80e30b  // sdot v11.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x4f81e30f  // sdot v15.4s, v24.16b, v1.4b[0]\n"
      ".inst 0x4f82e313  // sdot v19.4s, v24.16b, v2.4b[0]\n"
      ".inst 0x4f83e317  // sdot v23.4s, v24.16b, v3.4b[0]\n"
      "ldr q24, [x9, #0x50]\n"
      ".inst 0x4fa0e328  // sdot v8.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x4fa1e32c  // sdot v12.4s, v25.16b, v1.4b[1]\n"
      ".inst 0x4fa2e330  // sdot v16.4s, v25.16b, v2.4b[1]\n"
      ".inst 0x4fa3e334  // sdot v20.4s, v25.16b, v3.4b[1]\n"
      "ldr q25, [x9, #0x60]\n"
      ".inst 0x4fa0e309  // sdot v9.4s, v24.16b, v0.4b[1]\n"
      ".inst 0x4fa1e30d  // sdot v13.4s, v24.16b, v1.4b[1]\n"
      ".inst 0x4fa2e311  // sdot v17.4s, v24.16b, v2.4b[1]\n"
      ".inst 0x4fa3e315  // sdot v21.4s, v24.16b, v3.4b[1]\n"
      "ldr q24, [x9, #0x70]\n"
      ".inst 0x4fa0e32a  // sdot v10.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x4fa1e32e  // sdot v14.4s, v25.16b, v1.4b[1]\n"
      ".inst 0x4fa2e332  // sdot v18.4s, v25.16b, v2.4b[1]\n"
      ".inst 0x4fa3e336  // sdot v22.4s, v25.16b, v3.4b[1]\n"
      "ldr q25, [x9, #0x80]\n"
      ".inst 0x4fa0e30b  // sdot v11.4s, v24.16b, v0.4b[1]\n"
      ".inst 0x4fa1e30f  // sdot v15.4s, v24.16b, v1.4b[1]\n"
      ".inst 0x4fa2e313  // sdot v19.4s, v24.16b, v2.4b[1]\n"
      ".inst 0x4fa3e317  // sdot v23.4s, v24.16b, v3.4b[1]\n"
      "ldr q24, [x9, #0x90]\n"
      ".inst 0x4f80eb28  // sdot v8.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x4f81eb2c  // sdot v12.4s, v25.16b, v1.4b[2]\n"
      ".inst 0x4f82eb30  // sdot v16.4s, v25.16b, v2.4b[2]\n"
      ".inst 0x4f83eb34  // sdot v20.4s, v25.16b, v3.4b[2]\n"
      "ldr q25, [x9, #0xa0]\n"
      ".inst 0x4f80eb09  // sdot v9.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4f81eb0d  // sdot v13.4s, v24.16b, v1.4b[2]\n"
      ".inst 0x4f82eb11  // sdot v17.4s, v24.16b, v2.4b[2]\n"
      ".inst 0x4f83eb15  // sdot v21.4s, v24.16b, v3.4b[2]\n"
      "ldr q24, [x9, #0xb0]\n"
      ".inst 0x4f80eb2a  // sdot v10.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x4f81eb2e  // sdot v14.4s, v25.16b, v1.4b[2]\n"
      ".inst 0x4f82eb32  // sdot v18.4s, v25.16b, v2.4b[2]\n"
      ".inst 0x4f83eb36  // sdot v22.4s, v25.16b, v3.4b[2]\n"
      "ldr q25, [x9, #0xc0]\n"
      ".inst 0x4f80eb0b  // sdot v11.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4f81eb0f  // sdot v15.4s, v24.16b, v1.4b[2]\n"
      ".inst 0x4f82eb13  // sdot v19.4s, v24.16b, v2.4b[2]\n"
      ".inst 0x4f83eb17  // sdot v23.4s, v24.16b, v3.4b[2]\n"
      "ldr q24, [x9, #0xd0]\n"
      ".inst 0x4fa0eb28  // sdot v8.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb2c  // sdot v12.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb30  // sdot v16.4s, v25.16b, v2.4b[3]\n"
      ".inst 0x4fa3eb34  // sdot v20.4s, v25.16b, v3.4b[3]\n"
      "ldr q25, [x9, #0xe0]\n"
      ".inst 0x4fa0eb09  // sdot v9.4s, v24.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb0d  // sdot v13.4s, v24.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb11  // sdot v17.4s, v24.16b, v2.4b[3]\n"
      ".inst 0x4fa3eb15  // sdot v21.4s, v24.16b, v3.4b[3]\n"
      "ldr q24, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4fa0eb2a  // sdot v10.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb2e  // sdot v14.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb32  // sdot v18.4s, v25.16b, v2.4b[3]\n"
      ".inst 0x4fa3eb36  // sdot v22.4s, v25.16b, v3.4b[3]\n"
      ".inst 0x4fa0eb0b  // sdot v11.4s, v24.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb0f  // sdot v15.4s, v24.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb13  // sdot v19.4s, v24.16b, v2.4b[3]\n"
      ".inst 0x4fa3eb17  // sdot v23.4s, v24.16b, v3.4b[3]\n"
      "90:"  // Height 4: Multiply loop: Main loop skip
      "cbz x27, 95f\n"
      "cmp x27, #0x4\n"
      "blt 92f\n"
      "91:"  // Height 4: Multiply loop: Odd block loop
      "ldr s29, [x26], #0x4\n"
      "ldr s28, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "cmp x27, #0x4\n"
      "ldr s27, [x24], #0x4\n"
      "ldr s26, [x23], #0x4\n"
      "ldr q25, [x9, #0x0]\n"
      "ldr q24, [x9, #0x10]\n"
      ".inst 0x4f9de328  // sdot v8.4s, v25.16b, v29.4b[0]\n"
      ".inst 0x4f9ce32c  // sdot v12.4s, v25.16b, v28.4b[0]\n"
      ".inst 0x4f9be330  // sdot v16.4s, v25.16b, v27.4b[0]\n"
      ".inst 0x4f9ae334  // sdot v20.4s, v25.16b, v26.4b[0]\n"
      "ldr q25, [x9, #0x20]\n"
      ".inst 0x4f9de309  // sdot v9.4s, v24.16b, v29.4b[0]\n"
      ".inst 0x4f9ce30d  // sdot v13.4s, v24.16b, v28.4b[0]\n"
      ".inst 0x4f9be311  // sdot v17.4s, v24.16b, v27.4b[0]\n"
      ".inst 0x4f9ae315  // sdot v21.4s, v24.16b, v26.4b[0]\n"
      "ldr q24, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      ".inst 0x4f9de32a  // sdot v10.4s, v25.16b, v29.4b[0]\n"
      ".inst 0x4f9ce32e  // sdot v14.4s, v25.16b, v28.4b[0]\n"
      ".inst 0x4f9be332  // sdot v18.4s, v25.16b, v27.4b[0]\n"
      ".inst 0x4f9ae336  // sdot v22.4s, v25.16b, v26.4b[0]\n"
      ".inst 0x4f9de30b  // sdot v11.4s, v24.16b, v29.4b[0]\n"
      ".inst 0x4f9ce30f  // sdot v15.4s, v24.16b, v28.4b[0]\n"
      ".inst 0x4f9be313  // sdot v19.4s, v24.16b, v27.4b[0]\n"
      ".inst 0x4f9ae317  // sdot v23.4s, v24.16b, v26.4b[0]\n"
      "bge 91b\n"
      "92:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x27, 95f\n"
      "tbz x27, #1, 93f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "tbz x27, #0, 94f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "ld1 { v3.b }[2], [x23]\n"
      "b 94f\n"
      "93:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "ldr b3, [x23, #0x0]\n"
      "94:"  // Height 4: Multiply loop: Ragged operand read: Done
      "ldr q25, [x9, #0x0]\n"
      "ldr q24, [x9, #0x10]\n"
      ".inst 0x4f80e328  // sdot v8.4s, v25.16b, v0.4b[0]\n"
      ".inst 0x4f81e32c  // sdot v12.4s, v25.16b, v1.4b[0]\n"
      ".inst 0x4f82e330  // sdot v16.4s, v25.16b, v2.4b[0]\n"
      ".inst 0x4f83e334  // sdot v20.4s, v25.16b, v3.4b[0]\n"
      "ldr q25, [x9, #0x20]\n"
      ".inst 0x4f80e309  // sdot v9.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x4f81e30d  // sdot v13.4s, v24.16b, v1.4b[0]\n"
      ".inst 0x4f82e311  // sdot v17.4s, v24.16b, v2.4b[0]\n"
      ".inst 0x4f83e315  // sdot v21.4s, v24.16b, v3.4b[0]\n"
      "ldr q24, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      ".inst 0x4f80e32a  // sdot v10.4s, v25.16b, v0.4b[0]\n"
      ".inst 0x4f81e32e  // sdot v14.4s, v25.16b, v1.4b[0]\n"
      ".inst 0x4f82e332  // sdot v18.4s, v25.16b, v2.4b[0]\n"
      ".inst 0x4f83e336  // sdot v22.4s, v25.16b, v3.4b[0]\n"
      ".inst 0x4f80e30b  // sdot v11.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x4f81e30f  // sdot v15.4s, v24.16b, v1.4b[0]\n"
      ".inst 0x4f82e313  // sdot v19.4s, v24.16b, v2.4b[0]\n"
      ".inst 0x4f83e317  // sdot v23.4s, v24.16b, v3.4b[0]\n"
      "95:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 85b\n"
      "ldr q27, [x14, #0x0]\n"
      "ldr q26, [x14, #0x10]\n"
      "add v8.4s, v8.4s, v27.4s\n"
      "add v9.4s, v9.4s, v26.4s\n"
      "ldr q25, [x14, #0x20]\n"
      "ldr q24, [x14, #0x30]\n"
      "add v10.4s, v10.4s, v25.4s\n"
      "add v11.4s, v11.4s, v24.4s\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x11, x20\n"
      "add x24, x25, x20\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "add x23, x24, x20\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add v12.4s, v12.4s, v27.4s\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "add v13.4s, v13.4s, v26.4s\n"
      "add v14.4s, v14.4s, v25.4s\n"
      "add x14, x14, #0x40\n"
      "add v15.4s, v15.4s, v24.4s\n"
      "add v16.4s, v16.4s, v27.4s\n"
      "add v17.4s, v17.4s, v26.4s\n"
      "add v18.4s, v18.4s, v25.4s\n"
      "add v19.4s, v19.4s, v24.4s\n"
      "add v20.4s, v20.4s, v27.4s\n"
      "add v21.4s, v21.4s, v26.4s\n"
      "add v22.4s, v22.4s, v25.4s\n"
      "add v23.4s, v23.4s, v24.4s\n"
      "tbz %x[flags], #4, 96f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 97f\n"
      "96:"  // Height 4: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "97:"  // Height 4: parameters loaded
      "sqrdmulh v8.4s, v8.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "sqrdmulh v12.4s, v12.4s, v4.4s\n"
      "sqrdmulh v13.4s, v13.4s, v5.4s\n"
      "sqrdmulh v14.4s, v14.4s, v6.4s\n"
      "sqrdmulh v15.4s, v15.4s, v7.4s\n"
      "sqrdmulh v16.4s, v16.4s, v4.4s\n"
      "sqrdmulh v17.4s, v17.4s, v5.4s\n"
      "sqrdmulh v18.4s, v18.4s, v6.4s\n"
      "sqrdmulh v19.4s, v19.4s, v7.4s\n"
      "sqrdmulh v20.4s, v20.4s, v4.4s\n"
      "sqrdmulh v21.4s, v21.4s, v5.4s\n"
      "sqrdmulh v22.4s, v22.4s, v6.4s\n"
      "sqrdmulh v23.4s, v23.4s, v7.4s\n"
      "tbz %x[flags], #5, 98f\n"
      "and v27.16b, v8.16b, v0.16b\n"
      "and v26.16b, v9.16b, v1.16b\n"
      "and v25.16b, v10.16b, v2.16b\n"
      "and v24.16b, v11.16b, v3.16b\n"
      "sshr v27.4s, v27.4s, #0x1f\n"
      "sshr v26.4s, v26.4s, #0x1f\n"
      "sshr v25.4s, v25.4s, #0x1f\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sqadd v8.4s, v8.4s, v27.4s\n"
      "sqadd v9.4s, v9.4s, v26.4s\n"
      "sqadd v10.4s, v10.4s, v25.4s\n"
      "sqadd v11.4s, v11.4s, v24.4s\n"
      "and v27.16b, v12.16b, v0.16b\n"
      "and v26.16b, v13.16b, v1.16b\n"
      "and v25.16b, v14.16b, v2.16b\n"
      "and v24.16b, v15.16b, v3.16b\n"
      "sshr v27.4s, v27.4s, #0x1f\n"
      "sshr v26.4s, v26.4s, #0x1f\n"
      "sshr v25.4s, v25.4s, #0x1f\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sqadd v12.4s, v12.4s, v27.4s\n"
      "sqadd v13.4s, v13.4s, v26.4s\n"
      "sqadd v14.4s, v14.4s, v25.4s\n"
      "sqadd v15.4s, v15.4s, v24.4s\n"
      "and v27.16b, v16.16b, v0.16b\n"
      "and v26.16b, v17.16b, v1.16b\n"
      "and v25.16b, v18.16b, v2.16b\n"
      "and v24.16b, v19.16b, v3.16b\n"
      "sshr v27.4s, v27.4s, #0x1f\n"
      "sshr v26.4s, v26.4s, #0x1f\n"
      "sshr v25.4s, v25.4s, #0x1f\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v27.4s\n"
      "sqadd v17.4s, v17.4s, v26.4s\n"
      "sqadd v18.4s, v18.4s, v25.4s\n"
      "sqadd v19.4s, v19.4s, v24.4s\n"
      "and v27.16b, v20.16b, v0.16b\n"
      "and v26.16b, v21.16b, v1.16b\n"
      "and v25.16b, v22.16b, v2.16b\n"
      "and v24.16b, v23.16b, v3.16b\n"
      "sshr v27.4s, v27.4s, #0x1f\n"
      "sshr v26.4s, v26.4s, #0x1f\n"
      "sshr v25.4s, v25.4s, #0x1f\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sqadd v20.4s, v20.4s, v27.4s\n"
      "sqadd v21.4s, v21.4s, v26.4s\n"
      "sqadd v22.4s, v22.4s, v25.4s\n"
      "sqadd v23.4s, v23.4s, v24.4s\n"
      "98:"  // Height 4: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v26.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v25.4s }, [x20]\n"
      "srshl v12.4s, v12.4s, v0.4s\n"
      "srshl v13.4s, v13.4s, v1.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v24.4s }, [x20]\n"
      "srshl v14.4s, v14.4s, v2.4s\n"
      "srshl v15.4s, v15.4s, v3.4s\n"
      "cmp x10, #0x10\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v1.4s\n"
      "srshl v22.4s, v22.4s, v2.4s\n"
      "srshl v23.4s, v23.4s, v3.4s\n"
      "add v8.4s, v8.4s, v26.4s\n"
      "add v9.4s, v9.4s, v26.4s\n"
      "add v10.4s, v10.4s, v26.4s\n"
      "add v11.4s, v11.4s, v26.4s\n"
      "add v12.4s, v12.4s, v26.4s\n"
      "add v13.4s, v13.4s, v26.4s\n"
      "add v14.4s, v14.4s, v26.4s\n"
      "add v15.4s, v15.4s, v26.4s\n"
      "add v16.4s, v16.4s, v26.4s\n"
      "add v17.4s, v17.4s, v26.4s\n"
      "add v18.4s, v18.4s, v26.4s\n"
      "add v19.4s, v19.4s, v26.4s\n"
      "add v20.4s, v20.4s, v26.4s\n"
      "add v21.4s, v21.4s, v26.4s\n"
      "add v22.4s, v22.4s, v26.4s\n"
      "add v23.4s, v23.4s, v26.4s\n"
      "smin v8.4s, v8.4s, v25.4s\n"
      "smin v9.4s, v9.4s, v25.4s\n"
      "smin v10.4s, v10.4s, v25.4s\n"
      "smin v11.4s, v11.4s, v25.4s\n"
      "smin v12.4s, v12.4s, v25.4s\n"
      "smin v13.4s, v13.4s, v25.4s\n"
      "smin v14.4s, v14.4s, v25.4s\n"
      "smin v15.4s, v15.4s, v25.4s\n"
      "smin v16.4s, v16.4s, v25.4s\n"
      "smin v17.4s, v17.4s, v25.4s\n"
      "smin v18.4s, v18.4s, v25.4s\n"
      "smin v19.4s, v19.4s, v25.4s\n"
      "smin v20.4s, v20.4s, v25.4s\n"
      "smin v21.4s, v21.4s, v25.4s\n"
      "smin v22.4s, v22.4s, v25.4s\n"
      "smin v23.4s, v23.4s, v25.4s\n"
      "smax v8.4s, v8.4s, v24.4s\n"
      "smax v9.4s, v9.4s, v24.4s\n"
      "smax v10.4s, v10.4s, v24.4s\n"
      "smax v11.4s, v11.4s, v24.4s\n"
      "smax v12.4s, v12.4s, v24.4s\n"
      "smax v13.4s, v13.4s, v24.4s\n"
      "smax v14.4s, v14.4s, v24.4s\n"
      "smax v15.4s, v15.4s, v24.4s\n"
      "smax v16.4s, v16.4s, v24.4s\n"
      "smax v17.4s, v17.4s, v24.4s\n"
      "smax v18.4s, v18.4s, v24.4s\n"
      "smax v19.4s, v19.4s, v24.4s\n"
      "smax v20.4s, v20.4s, v24.4s\n"
      "smax v21.4s, v21.4s, v24.4s\n"
      "smax v22.4s, v22.4s, v24.4s\n"
      "smax v23.4s, v23.4s, v24.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v25.8h, v10.8h, v11.8h\n"
      "uzp1 v12.8h, v12.8h, v13.8h\n"
      "uzp1 v24.8h, v14.8h, v15.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v18.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "uzp1 v17.8h, v22.8h, v23.8h\n"
      "uzp1 v8.16b, v8.16b, v25.16b\n"
      "uzp1 v12.16b, v12.16b, v24.16b\n"
      "uzp1 v16.16b, v16.16b, v18.16b\n"
      "uzp1 v20.16b, v20.16b, v17.16b\n"
      "bge 107f\n"
      "tbz x10, #3, 102f\n"
      "str d8, [x11], #0x8\n"
      "str d12, [x25], #0x8\n"
      "str d16, [x24], #0x8\n"
      "str d20, [x23], #0x8\n"
      "tbz x10, #2, 100f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "st1 { v12.s }[2], [x25], #0x4\n"
      "st1 { v16.s }[2], [x24], #0x4\n"
      "st1 { v20.s }[2], [x23], #0x4\n"
      "tbz x10, #1, 99f\n"
      "st1 { v8.h }[6], [x11], #0x2\n"
      "st1 { v12.h }[6], [x25], #0x2\n"
      "st1 { v16.h }[6], [x24], #0x2\n"
      "st1 { v20.h }[6], [x23], #0x2\n"
      "tbz x10, #0, 106f\n"
      "st1 { v8.b }[14], [x11]\n"
      "st1 { v12.b }[14], [x25]\n"
      "st1 { v16.b }[14], [x24]\n"
      "st1 { v20.b }[14], [x23]\n"
      "b 106f\n"
      "99:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x10, #0, 106f\n"
      "st1 { v8.b }[12], [x11]\n"
      "st1 { v12.b }[12], [x25]\n"
      "st1 { v16.b }[12], [x24]\n"
      "st1 { v20.b }[12], [x23]\n"
      "b 106f\n"
      "100:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x10, #1, 101f\n"
      "st1 { v8.h }[4], [x11], #0x2\n"
      "st1 { v12.h }[4], [x25], #0x2\n"
      "st1 { v16.h }[4], [x24], #0x2\n"
      "st1 { v20.h }[4], [x23], #0x2\n"
      "tbz x10, #0, 106f\n"
      "st1 { v8.b }[10], [x11]\n"
      "st1 { v12.b }[10], [x25]\n"
      "st1 { v16.b }[10], [x24]\n"
      "st1 { v20.b }[10], [x23]\n"
      "b 106f\n"
      "101:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x10, #0, 106f\n"
      "st1 { v8.b }[8], [x11]\n"
      "st1 { v12.b }[8], [x25]\n"
      "st1 { v16.b }[8], [x24]\n"
      "st1 { v20.b }[8], [x23]\n"
      "b 106f\n"
      "102:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x10, #2, 104f\n"
      "str s8, [x11], #0x4\n"
      "str s12, [x25], #0x4\n"
      "str s16, [x24], #0x4\n"
      "str s20, [x23], #0x4\n"
      "tbz x10, #1, 103f\n"
      "st1 { v8.h }[2], [x11], #0x2\n"
      "st1 { v12.h }[2], [x25], #0x2\n"
      "st1 { v16.h }[2], [x24], #0x2\n"
      "st1 { v20.h }[2], [x23], #0x2\n"
      "tbz x10, #0, 106f\n"
      "st1 { v8.b }[6], [x11]\n"
      "st1 { v12.b }[6], [x25]\n"
      "st1 { v16.b }[6], [x24]\n"
      "st1 { v20.b }[6], [x23]\n"
      "b 106f\n"
      "103:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x10, #0, 106f\n"
      "st1 { v8.b }[4], [x11]\n"
      "st1 { v12.b }[4], [x25]\n"
      "st1 { v16.b }[4], [x24]\n"
      "st1 { v20.b }[4], [x23]\n"
      "b 106f\n"
      "104:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x10, #1, 105f\n"
      "str h8, [x11], #0x2\n"
      "str h12, [x25], #0x2\n"
      "str h16, [x24], #0x2\n"
      "str h20, [x23], #0x2\n"
      "tbz x10, #0, 106f\n"
      "st1 { v8.b }[2], [x11]\n"
      "st1 { v12.b }[2], [x25]\n"
      "st1 { v16.b }[2], [x24]\n"
      "st1 { v20.b }[2], [x23]\n"
      "b 106f\n"
      "105:"  // Height 4: Partial direct writeback: partial_1_0
      "str b8, [x11, #0x0]\n"
      "str b12, [x25, #0x0]\n"
      "str b16, [x24, #0x0]\n"
      "str b20, [x23, #0x0]\n"
      "106:"  // Height 4: Partial direct writeback: Done
      "b 108f\n"
      "107:"  // Height 4: Full writeback
      "str q8, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q12, [x25, #0x0]\n"
      "str q16, [x24, #0x0]\n"
      "str q20, [x23, #0x0]\n"
      "108:"  // Height 4: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 83b\n"
      "b 164f\n"
      "109:"  // Height 5
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "110:"  // Height 5: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "111:"  // Height 5: setup done
      "mov x28, #0x0\n"
      "112:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 113f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 114f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 114f\n"
      "113:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "114:"  // Height 5: input setup done
      "cmp x27, #0x10\n"
      "blt 117f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q6, [x9, #0x0]\n"
      "ldr q7, [x9, #0x10]\n"
      "blt 116f\n"
      "115:"  // Height 5: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q29, [x9, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "add x22, x22, #0x10\n"
      "cmp x27, #0x20\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q28, [x9, #0x30]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f80e3aa  // sdot v10.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f81e3ae  // sdot v14.4s, v29.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f82e3b2  // sdot v18.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f83e3b6  // sdot v22.4s, v29.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f84e3ba  // sdot v26.4s, v29.16b, v4.4b[0]\n"
      "ldr q29, [x9, #0x40]\n"
      ".inst 0x4f80e38b  // sdot v11.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f81e38f  // sdot v15.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f82e393  // sdot v19.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f83e397  // sdot v23.4s, v28.16b, v3.4b[0]\n"
      ".inst 0x4f84e39b  // sdot v27.4s, v28.16b, v4.4b[0]\n"
      "ldr q28, [x9, #0x50]\n"
      ".inst 0x4fa0e3a8  // sdot v8.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x4fa1e3ac  // sdot v12.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x4fa2e3b0  // sdot v16.4s, v29.16b, v2.4b[1]\n"
      ".inst 0x4fa3e3b4  // sdot v20.4s, v29.16b, v3.4b[1]\n"
      ".inst 0x4fa4e3b8  // sdot v24.4s, v29.16b, v4.4b[1]\n"
      "ldr q29, [x9, #0x60]\n"
      ".inst 0x4fa0e389  // sdot v9.4s, v28.16b, v0.4b[1]\n"
      ".inst 0x4fa1e38d  // sdot v13.4s, v28.16b, v1.4b[1]\n"
      ".inst 0x4fa2e391  // sdot v17.4s, v28.16b, v2.4b[1]\n"
      ".inst 0x4fa3e395  // sdot v21.4s, v28.16b, v3.4b[1]\n"
      ".inst 0x4fa4e399  // sdot v25.4s, v28.16b, v4.4b[1]\n"
      "ldr q28, [x9, #0x70]\n"
      ".inst 0x4fa0e3aa  // sdot v10.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x4fa1e3ae  // sdot v14.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x4fa2e3b2  // sdot v18.4s, v29.16b, v2.4b[1]\n"
      ".inst 0x4fa3e3b6  // sdot v22.4s, v29.16b, v3.4b[1]\n"
      ".inst 0x4fa4e3ba  // sdot v26.4s, v29.16b, v4.4b[1]\n"
      "ldr q29, [x9, #0x80]\n"
      ".inst 0x4fa0e38b  // sdot v11.4s, v28.16b, v0.4b[1]\n"
      ".inst 0x4fa1e38f  // sdot v15.4s, v28.16b, v1.4b[1]\n"
      ".inst 0x4fa2e393  // sdot v19.4s, v28.16b, v2.4b[1]\n"
      ".inst 0x4fa3e397  // sdot v23.4s, v28.16b, v3.4b[1]\n"
      ".inst 0x4fa4e39b  // sdot v27.4s, v28.16b, v4.4b[1]\n"
      "ldr q28, [x9, #0x90]\n"
      ".inst 0x4f80eba8  // sdot v8.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x4f81ebac  // sdot v12.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x4f82ebb0  // sdot v16.4s, v29.16b, v2.4b[2]\n"
      ".inst 0x4f83ebb4  // sdot v20.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x4f84ebb8  // sdot v24.4s, v29.16b, v4.4b[2]\n"
      "ldr q29, [x9, #0xa0]\n"
      ".inst 0x4f80eb89  // sdot v9.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f81eb8d  // sdot v13.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4f82eb91  // sdot v17.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x4f83eb95  // sdot v21.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x4f84eb99  // sdot v25.4s, v28.16b, v4.4b[2]\n"
      "ldr q28, [x9, #0xb0]\n"
      ".inst 0x4f80ebaa  // sdot v10.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x4f81ebae  // sdot v14.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x4f82ebb2  // sdot v18.4s, v29.16b, v2.4b[2]\n"
      ".inst 0x4f83ebb6  // sdot v22.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x4f84ebba  // sdot v26.4s, v29.16b, v4.4b[2]\n"
      "ldr q29, [x9, #0xc0]\n"
      ".inst 0x4f80eb8b  // sdot v11.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f81eb8f  // sdot v15.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4f82eb93  // sdot v19.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x4f83eb97  // sdot v23.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x4f84eb9b  // sdot v27.4s, v28.16b, v4.4b[2]\n"
      "ldr q28, [x9, #0xd0]\n"
      ".inst 0x4fa0eba8  // sdot v8.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x4fa1ebac  // sdot v12.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x4fa2ebb0  // sdot v16.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x4fa3ebb4  // sdot v20.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x4fa4ebb8  // sdot v24.4s, v29.16b, v4.4b[3]\n"
      "ldr q29, [x9, #0xe0]\n"
      ".inst 0x4fa0eb89  // sdot v9.4s, v28.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb8d  // sdot v13.4s, v28.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb91  // sdot v17.4s, v28.16b, v2.4b[3]\n"
      ".inst 0x4fa3eb95  // sdot v21.4s, v28.16b, v3.4b[3]\n"
      ".inst 0x4fa4eb99  // sdot v25.4s, v28.16b, v4.4b[3]\n"
      "ldr q28, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4fa0ebaa  // sdot v10.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x4fa1ebae  // sdot v14.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x4fa2ebb2  // sdot v18.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x4fa3ebb6  // sdot v22.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x4fa4ebba  // sdot v26.4s, v29.16b, v4.4b[3]\n"
      "ldr q6, [x9, #0x0]\n"
      ".inst 0x4fa0eb8b  // sdot v11.4s, v28.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4fa1eb8f  // sdot v15.4s, v28.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4fa2eb93  // sdot v19.4s, v28.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      ".inst 0x4fa3eb97  // sdot v23.4s, v28.16b, v3.4b[3]\n"
      "ldr q3, [x23, #0x0]\n"
      ".inst 0x4fa4eb9b  // sdot v27.4s, v28.16b, v4.4b[3]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q7, [x9, #0x10]\n"
      "bge 115b\n"
      "116:"  // Height 5: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q29, [x9, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q28, [x9, #0x30]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f80e3aa  // sdot v10.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f81e3ae  // sdot v14.4s, v29.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f82e3b2  // sdot v18.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f83e3b6  // sdot v22.4s, v29.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f84e3ba  // sdot v26.4s, v29.16b, v4.4b[0]\n"
      "ldr q29, [x9, #0x40]\n"
      ".inst 0x4f80e38b  // sdot v11.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f81e38f  // sdot v15.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f82e393  // sdot v19.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f83e397  // sdot v23.4s, v28.16b, v3.4b[0]\n"
      ".inst 0x4f84e39b  // sdot v27.4s, v28.16b, v4.4b[0]\n"
      "ldr q28, [x9, #0x50]\n"
      ".inst 0x4fa0e3a8  // sdot v8.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x4fa1e3ac  // sdot v12.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x4fa2e3b0  // sdot v16.4s, v29.16b, v2.4b[1]\n"
      ".inst 0x4fa3e3b4  // sdot v20.4s, v29.16b, v3.4b[1]\n"
      ".inst 0x4fa4e3b8  // sdot v24.4s, v29.16b, v4.4b[1]\n"
      "ldr q29, [x9, #0x60]\n"
      ".inst 0x4fa0e389  // sdot v9.4s, v28.16b, v0.4b[1]\n"
      ".inst 0x4fa1e38d  // sdot v13.4s, v28.16b, v1.4b[1]\n"
      ".inst 0x4fa2e391  // sdot v17.4s, v28.16b, v2.4b[1]\n"
      ".inst 0x4fa3e395  // sdot v21.4s, v28.16b, v3.4b[1]\n"
      ".inst 0x4fa4e399  // sdot v25.4s, v28.16b, v4.4b[1]\n"
      "ldr q28, [x9, #0x70]\n"
      ".inst 0x4fa0e3aa  // sdot v10.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x4fa1e3ae  // sdot v14.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x4fa2e3b2  // sdot v18.4s, v29.16b, v2.4b[1]\n"
      ".inst 0x4fa3e3b6  // sdot v22.4s, v29.16b, v3.4b[1]\n"
      ".inst 0x4fa4e3ba  // sdot v26.4s, v29.16b, v4.4b[1]\n"
      "ldr q29, [x9, #0x80]\n"
      ".inst 0x4fa0e38b  // sdot v11.4s, v28.16b, v0.4b[1]\n"
      ".inst 0x4fa1e38f  // sdot v15.4s, v28.16b, v1.4b[1]\n"
      ".inst 0x4fa2e393  // sdot v19.4s, v28.16b, v2.4b[1]\n"
      ".inst 0x4fa3e397  // sdot v23.4s, v28.16b, v3.4b[1]\n"
      ".inst 0x4fa4e39b  // sdot v27.4s, v28.16b, v4.4b[1]\n"
      "ldr q28, [x9, #0x90]\n"
      ".inst 0x4f80eba8  // sdot v8.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x4f81ebac  // sdot v12.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x4f82ebb0  // sdot v16.4s, v29.16b, v2.4b[2]\n"
      ".inst 0x4f83ebb4  // sdot v20.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x4f84ebb8  // sdot v24.4s, v29.16b, v4.4b[2]\n"
      "ldr q29, [x9, #0xa0]\n"
      ".inst 0x4f80eb89  // sdot v9.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f81eb8d  // sdot v13.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4f82eb91  // sdot v17.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x4f83eb95  // sdot v21.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x4f84eb99  // sdot v25.4s, v28.16b, v4.4b[2]\n"
      "ldr q28, [x9, #0xb0]\n"
      ".inst 0x4f80ebaa  // sdot v10.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x4f81ebae  // sdot v14.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x4f82ebb2  // sdot v18.4s, v29.16b, v2.4b[2]\n"
      ".inst 0x4f83ebb6  // sdot v22.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x4f84ebba  // sdot v26.4s, v29.16b, v4.4b[2]\n"
      "ldr q29, [x9, #0xc0]\n"
      ".inst 0x4f80eb8b  // sdot v11.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f81eb8f  // sdot v15.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4f82eb93  // sdot v19.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x4f83eb97  // sdot v23.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x4f84eb9b  // sdot v27.4s, v28.16b, v4.4b[2]\n"
      "ldr q28, [x9, #0xd0]\n"
      ".inst 0x4fa0eba8  // sdot v8.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x4fa1ebac  // sdot v12.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x4fa2ebb0  // sdot v16.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x4fa3ebb4  // sdot v20.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x4fa4ebb8  // sdot v24.4s, v29.16b, v4.4b[3]\n"
      "ldr q29, [x9, #0xe0]\n"
      ".inst 0x4fa0eb89  // sdot v9.4s, v28.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb8d  // sdot v13.4s, v28.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb91  // sdot v17.4s, v28.16b, v2.4b[3]\n"
      ".inst 0x4fa3eb95  // sdot v21.4s, v28.16b, v3.4b[3]\n"
      ".inst 0x4fa4eb99  // sdot v25.4s, v28.16b, v4.4b[3]\n"
      "ldr q28, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4fa0ebaa  // sdot v10.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x4fa1ebae  // sdot v14.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x4fa2ebb2  // sdot v18.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x4fa3ebb6  // sdot v22.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x4fa4ebba  // sdot v26.4s, v29.16b, v4.4b[3]\n"
      ".inst 0x4fa0eb8b  // sdot v11.4s, v28.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb8f  // sdot v15.4s, v28.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb93  // sdot v19.4s, v28.16b, v2.4b[3]\n"
      ".inst 0x4fa3eb97  // sdot v23.4s, v28.16b, v3.4b[3]\n"
      ".inst 0x4fa4eb9b  // sdot v27.4s, v28.16b, v4.4b[3]\n"
      "117:"  // Height 5: Multiply loop: Main loop skip
      "cbz x27, 122f\n"
      "cmp x27, #0x4\n"
      "blt 119f\n"
      "118:"  // Height 5: Multiply loop: Odd block loop
      "ldr s2, [x26], #0x4\n"
      "ldr s1, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "cmp x27, #0x4\n"
      "ldr s0, [x24], #0x4\n"
      "ldr s31, [x23], #0x4\n"
      "ldr s30, [x22], #0x4\n"
      "ldr q29, [x9, #0x0]\n"
      ".inst 0x4f82e3a8  // sdot v8.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f81e3ac  // sdot v12.4s, v29.16b, v1.4b[0]\n"
      "ldr q28, [x9, #0x10]\n"
      ".inst 0x4f80e3b0  // sdot v16.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f9fe3b4  // sdot v20.4s, v29.16b, v31.4b[0]\n"
      ".inst 0x4f9ee3b8  // sdot v24.4s, v29.16b, v30.4b[0]\n"
      "ldr q29, [x9, #0x20]\n"
      ".inst 0x4f82e389  // sdot v9.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f81e38d  // sdot v13.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f80e391  // sdot v17.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f9fe395  // sdot v21.4s, v28.16b, v31.4b[0]\n"
      ".inst 0x4f9ee399  // sdot v25.4s, v28.16b, v30.4b[0]\n"
      "ldr q28, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      ".inst 0x4f82e3aa  // sdot v10.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f81e3ae  // sdot v14.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x4f80e3b2  // sdot v18.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f9fe3b6  // sdot v22.4s, v29.16b, v31.4b[0]\n"
      ".inst 0x4f9ee3ba  // sdot v26.4s, v29.16b, v30.4b[0]\n"
      ".inst 0x4f82e38b  // sdot v11.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f81e38f  // sdot v15.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f80e393  // sdot v19.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f9fe397  // sdot v23.4s, v28.16b, v31.4b[0]\n"
      ".inst 0x4f9ee39b  // sdot v27.4s, v28.16b, v30.4b[0]\n"
      "bge 118b\n"
      "119:"  // Height 5: Multiply loop: Skip odd blocks
      "cbz x27, 122f\n"
      "tbz x27, #1, 120f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr h4, [x22], #0x2\n"
      "tbz x27, #0, 121f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "ld1 { v3.b }[2], [x23]\n"
      "ld1 { v4.b }[2], [x22]\n"
      "b 121f\n"
      "120:"  // Height 5: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "ldr b3, [x23, #0x0]\n"
      "ldr b4, [x22, #0x0]\n"
      "121:"  // Height 5: Multiply loop: Ragged operand read: Done
      "ldr q29, [x9, #0x0]\n"
      "ldr q28, [x9, #0x10]\n"
      ".inst 0x4f80e3a8  // sdot v8.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f81e3ac  // sdot v12.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x4f82e3b0  // sdot v16.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f83e3b4  // sdot v20.4s, v29.16b, v3.4b[0]\n"
      ".inst 0x4f84e3b8  // sdot v24.4s, v29.16b, v4.4b[0]\n"
      "ldr q29, [x9, #0x20]\n"
      ".inst 0x4f80e389  // sdot v9.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f81e38d  // sdot v13.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f82e391  // sdot v17.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f83e395  // sdot v21.4s, v28.16b, v3.4b[0]\n"
      ".inst 0x4f84e399  // sdot v25.4s, v28.16b, v4.4b[0]\n"
      "ldr q28, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      ".inst 0x4f80e3aa  // sdot v10.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f81e3ae  // sdot v14.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x4f82e3b2  // sdot v18.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f83e3b6  // sdot v22.4s, v29.16b, v3.4b[0]\n"
      ".inst 0x4f84e3ba  // sdot v26.4s, v29.16b, v4.4b[0]\n"
      ".inst 0x4f80e38b  // sdot v11.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f81e38f  // sdot v15.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f82e393  // sdot v19.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f83e397  // sdot v23.4s, v28.16b, v3.4b[0]\n"
      ".inst 0x4f84e39b  // sdot v27.4s, v28.16b, v4.4b[0]\n"
      "122:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 112b\n"
      "ldr q31, [x14, #0x0]\n"
      "ldr q30, [x14, #0x10]\n"
      "add v8.4s, v8.4s, v31.4s\n"
      "add v9.4s, v9.4s, v30.4s\n"
      "ldr q29, [x14, #0x20]\n"
      "ldr q28, [x14, #0x30]\n"
      "add v10.4s, v10.4s, v29.4s\n"
      "add v11.4s, v11.4s, v28.4s\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x11, x20\n"
      "add x24, x25, x20\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "add x23, x24, x20\n"
      "add x22, x23, x20\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "add v12.4s, v12.4s, v31.4s\n"
      "add v13.4s, v13.4s, v30.4s\n"
      "add v14.4s, v14.4s, v29.4s\n"
      "add v15.4s, v15.4s, v28.4s\n"
      "add x14, x14, #0x40\n"
      "add v16.4s, v16.4s, v31.4s\n"
      "add v17.4s, v17.4s, v30.4s\n"
      "add v18.4s, v18.4s, v29.4s\n"
      "add v19.4s, v19.4s, v28.4s\n"
      "add v20.4s, v20.4s, v31.4s\n"
      "add v21.4s, v21.4s, v30.4s\n"
      "add v22.4s, v22.4s, v29.4s\n"
      "add v23.4s, v23.4s, v28.4s\n"
      "add v24.4s, v24.4s, v31.4s\n"
      "add v25.4s, v25.4s, v30.4s\n"
      "add v26.4s, v26.4s, v29.4s\n"
      "add v27.4s, v27.4s, v28.4s\n"
      "tbz %x[flags], #4, 123f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 124f\n"
      "123:"  // Height 5: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "124:"  // Height 5: parameters loaded
      "sqrdmulh v8.4s, v8.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "sqrdmulh v12.4s, v12.4s, v4.4s\n"
      "sqrdmulh v13.4s, v13.4s, v5.4s\n"
      "sqrdmulh v14.4s, v14.4s, v6.4s\n"
      "sqrdmulh v15.4s, v15.4s, v7.4s\n"
      "sqrdmulh v16.4s, v16.4s, v4.4s\n"
      "sqrdmulh v17.4s, v17.4s, v5.4s\n"
      "sqrdmulh v18.4s, v18.4s, v6.4s\n"
      "sqrdmulh v19.4s, v19.4s, v7.4s\n"
      "sqrdmulh v20.4s, v20.4s, v4.4s\n"
      "sqrdmulh v21.4s, v21.4s, v5.4s\n"
      "sqrdmulh v22.4s, v22.4s, v6.4s\n"
      "sqrdmulh v23.4s, v23.4s, v7.4s\n"
      "sqrdmulh v24.4s, v24.4s, v4.4s\n"
      "sqrdmulh v25.4s, v25.4s, v5.4s\n"
      "sqrdmulh v26.4s, v26.4s, v6.4s\n"
      "sqrdmulh v27.4s, v27.4s, v7.4s\n"
      "tbz %x[flags], #5, 125f\n"
      "and v31.16b, v8.16b, v0.16b\n"
      "and v30.16b, v9.16b, v1.16b\n"
      "and v29.16b, v10.16b, v2.16b\n"
      "and v28.16b, v11.16b, v3.16b\n"
      "sshr v31.4s, v31.4s, #0x1f\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sqadd v8.4s, v8.4s, v31.4s\n"
      "sqadd v9.4s, v9.4s, v30.4s\n"
      "sqadd v10.4s, v10.4s, v29.4s\n"
      "sqadd v11.4s, v11.4s, v28.4s\n"
      "and v31.16b, v12.16b, v0.16b\n"
      "and v30.16b, v13.16b, v1.16b\n"
      "and v29.16b, v14.16b, v2.16b\n"
      "and v28.16b, v15.16b, v3.16b\n"
      "sshr v31.4s, v31.4s, #0x1f\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sqadd v12.4s, v12.4s, v31.4s\n"
      "sqadd v13.4s, v13.4s, v30.4s\n"
      "sqadd v14.4s, v14.4s, v29.4s\n"
      "sqadd v15.4s, v15.4s, v28.4s\n"
      "and v31.16b, v16.16b, v0.16b\n"
      "and v30.16b, v17.16b, v1.16b\n"
      "and v29.16b, v18.16b, v2.16b\n"
      "and v28.16b, v19.16b, v3.16b\n"
      "sshr v31.4s, v31.4s, #0x1f\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v31.4s\n"
      "sqadd v17.4s, v17.4s, v30.4s\n"
      "sqadd v18.4s, v18.4s, v29.4s\n"
      "sqadd v19.4s, v19.4s, v28.4s\n"
      "and v31.16b, v20.16b, v0.16b\n"
      "and v30.16b, v21.16b, v1.16b\n"
      "and v29.16b, v22.16b, v2.16b\n"
      "and v28.16b, v23.16b, v3.16b\n"
      "sshr v31.4s, v31.4s, #0x1f\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sqadd v20.4s, v20.4s, v31.4s\n"
      "sqadd v21.4s, v21.4s, v30.4s\n"
      "sqadd v22.4s, v22.4s, v29.4s\n"
      "sqadd v23.4s, v23.4s, v28.4s\n"
      "and v31.16b, v24.16b, v0.16b\n"
      "and v30.16b, v25.16b, v1.16b\n"
      "and v29.16b, v26.16b, v2.16b\n"
      "and v28.16b, v27.16b, v3.16b\n"
      "sshr v31.4s, v31.4s, #0x1f\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sqadd v24.4s, v24.4s, v31.4s\n"
      "sqadd v25.4s, v25.4s, v30.4s\n"
      "sqadd v26.4s, v26.4s, v29.4s\n"
      "sqadd v27.4s, v27.4s, v28.4s\n"
      "125:"  // Height 5: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v30.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v29.4s }, [x20]\n"
      "srshl v12.4s, v12.4s, v0.4s\n"
      "srshl v13.4s, v13.4s, v1.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v28.4s }, [x20]\n"
      "srshl v14.4s, v14.4s, v2.4s\n"
      "srshl v15.4s, v15.4s, v3.4s\n"
      "cmp x10, #0x10\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v1.4s\n"
      "srshl v22.4s, v22.4s, v2.4s\n"
      "srshl v23.4s, v23.4s, v3.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v1.4s\n"
      "srshl v26.4s, v26.4s, v2.4s\n"
      "srshl v27.4s, v27.4s, v3.4s\n"
      "add v8.4s, v8.4s, v30.4s\n"
      "add v9.4s, v9.4s, v30.4s\n"
      "add v10.4s, v10.4s, v30.4s\n"
      "add v11.4s, v11.4s, v30.4s\n"
      "add v12.4s, v12.4s, v30.4s\n"
      "add v13.4s, v13.4s, v30.4s\n"
      "add v14.4s, v14.4s, v30.4s\n"
      "add v15.4s, v15.4s, v30.4s\n"
      "add v16.4s, v16.4s, v30.4s\n"
      "add v17.4s, v17.4s, v30.4s\n"
      "add v18.4s, v18.4s, v30.4s\n"
      "add v19.4s, v19.4s, v30.4s\n"
      "add v20.4s, v20.4s, v30.4s\n"
      "add v21.4s, v21.4s, v30.4s\n"
      "add v22.4s, v22.4s, v30.4s\n"
      "add v23.4s, v23.4s, v30.4s\n"
      "add v24.4s, v24.4s, v30.4s\n"
      "add v25.4s, v25.4s, v30.4s\n"
      "add v26.4s, v26.4s, v30.4s\n"
      "add v27.4s, v27.4s, v30.4s\n"
      "smin v8.4s, v8.4s, v29.4s\n"
      "smin v9.4s, v9.4s, v29.4s\n"
      "smin v10.4s, v10.4s, v29.4s\n"
      "smin v11.4s, v11.4s, v29.4s\n"
      "smin v12.4s, v12.4s, v29.4s\n"
      "smin v13.4s, v13.4s, v29.4s\n"
      "smin v14.4s, v14.4s, v29.4s\n"
      "smin v15.4s, v15.4s, v29.4s\n"
      "smin v16.4s, v16.4s, v29.4s\n"
      "smin v17.4s, v17.4s, v29.4s\n"
      "smin v18.4s, v18.4s, v29.4s\n"
      "smin v19.4s, v19.4s, v29.4s\n"
      "smin v20.4s, v20.4s, v29.4s\n"
      "smin v21.4s, v21.4s, v29.4s\n"
      "smin v22.4s, v22.4s, v29.4s\n"
      "smin v23.4s, v23.4s, v29.4s\n"
      "smin v24.4s, v24.4s, v29.4s\n"
      "smin v25.4s, v25.4s, v29.4s\n"
      "smin v26.4s, v26.4s, v29.4s\n"
      "smin v27.4s, v27.4s, v29.4s\n"
      "smax v8.4s, v8.4s, v28.4s\n"
      "smax v9.4s, v9.4s, v28.4s\n"
      "smax v10.4s, v10.4s, v28.4s\n"
      "smax v11.4s, v11.4s, v28.4s\n"
      "smax v12.4s, v12.4s, v28.4s\n"
      "smax v13.4s, v13.4s, v28.4s\n"
      "smax v14.4s, v14.4s, v28.4s\n"
      "smax v15.4s, v15.4s, v28.4s\n"
      "smax v16.4s, v16.4s, v28.4s\n"
      "smax v17.4s, v17.4s, v28.4s\n"
      "smax v18.4s, v18.4s, v28.4s\n"
      "smax v19.4s, v19.4s, v28.4s\n"
      "smax v20.4s, v20.4s, v28.4s\n"
      "smax v21.4s, v21.4s, v28.4s\n"
      "smax v22.4s, v22.4s, v28.4s\n"
      "smax v23.4s, v23.4s, v28.4s\n"
      "smax v24.4s, v24.4s, v28.4s\n"
      "smax v25.4s, v25.4s, v28.4s\n"
      "smax v26.4s, v26.4s, v28.4s\n"
      "smax v27.4s, v27.4s, v28.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v29.8h, v10.8h, v11.8h\n"
      "uzp1 v12.8h, v12.8h, v13.8h\n"
      "uzp1 v28.8h, v14.8h, v15.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v19.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "uzp1 v18.8h, v22.8h, v23.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v17.8h, v26.8h, v27.8h\n"
      "uzp1 v8.16b, v8.16b, v29.16b\n"
      "uzp1 v12.16b, v12.16b, v28.16b\n"
      "uzp1 v16.16b, v16.16b, v19.16b\n"
      "uzp1 v20.16b, v20.16b, v18.16b\n"
      "uzp1 v24.16b, v24.16b, v17.16b\n"
      "bge 134f\n"
      "tbz x10, #3, 129f\n"
      "str d8, [x11], #0x8\n"
      "str d12, [x25], #0x8\n"
      "str d16, [x24], #0x8\n"
      "str d20, [x23], #0x8\n"
      "str d24, [x22], #0x8\n"
      "tbz x10, #2, 127f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "st1 { v12.s }[2], [x25], #0x4\n"
      "st1 { v16.s }[2], [x24], #0x4\n"
      "st1 { v20.s }[2], [x23], #0x4\n"
      "st1 { v24.s }[2], [x22], #0x4\n"
      "tbz x10, #1, 126f\n"
      "st1 { v8.h }[6], [x11], #0x2\n"
      "st1 { v12.h }[6], [x25], #0x2\n"
      "st1 { v16.h }[6], [x24], #0x2\n"
      "st1 { v20.h }[6], [x23], #0x2\n"
      "st1 { v24.h }[6], [x22], #0x2\n"
      "tbz x10, #0, 133f\n"
      "st1 { v8.b }[14], [x11]\n"
      "st1 { v12.b }[14], [x25]\n"
      "st1 { v16.b }[14], [x24]\n"
      "st1 { v20.b }[14], [x23]\n"
      "st1 { v24.b }[14], [x22]\n"
      "b 133f\n"
      "126:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x10, #0, 133f\n"
      "st1 { v8.b }[12], [x11]\n"
      "st1 { v12.b }[12], [x25]\n"
      "st1 { v16.b }[12], [x24]\n"
      "st1 { v20.b }[12], [x23]\n"
      "st1 { v24.b }[12], [x22]\n"
      "b 133f\n"
      "127:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x10, #1, 128f\n"
      "st1 { v8.h }[4], [x11], #0x2\n"
      "st1 { v12.h }[4], [x25], #0x2\n"
      "st1 { v16.h }[4], [x24], #0x2\n"
      "st1 { v20.h }[4], [x23], #0x2\n"
      "st1 { v24.h }[4], [x22], #0x2\n"
      "tbz x10, #0, 133f\n"
      "st1 { v8.b }[10], [x11]\n"
      "st1 { v12.b }[10], [x25]\n"
      "st1 { v16.b }[10], [x24]\n"
      "st1 { v20.b }[10], [x23]\n"
      "st1 { v24.b }[10], [x22]\n"
      "b 133f\n"
      "128:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x10, #0, 133f\n"
      "st1 { v8.b }[8], [x11]\n"
      "st1 { v12.b }[8], [x25]\n"
      "st1 { v16.b }[8], [x24]\n"
      "st1 { v20.b }[8], [x23]\n"
      "st1 { v24.b }[8], [x22]\n"
      "b 133f\n"
      "129:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x10, #2, 131f\n"
      "str s8, [x11], #0x4\n"
      "str s12, [x25], #0x4\n"
      "str s16, [x24], #0x4\n"
      "str s20, [x23], #0x4\n"
      "str s24, [x22], #0x4\n"
      "tbz x10, #1, 130f\n"
      "st1 { v8.h }[2], [x11], #0x2\n"
      "st1 { v12.h }[2], [x25], #0x2\n"
      "st1 { v16.h }[2], [x24], #0x2\n"
      "st1 { v20.h }[2], [x23], #0x2\n"
      "st1 { v24.h }[2], [x22], #0x2\n"
      "tbz x10, #0, 133f\n"
      "st1 { v8.b }[6], [x11]\n"
      "st1 { v12.b }[6], [x25]\n"
      "st1 { v16.b }[6], [x24]\n"
      "st1 { v20.b }[6], [x23]\n"
      "st1 { v24.b }[6], [x22]\n"
      "b 133f\n"
      "130:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x10, #0, 133f\n"
      "st1 { v8.b }[4], [x11]\n"
      "st1 { v12.b }[4], [x25]\n"
      "st1 { v16.b }[4], [x24]\n"
      "st1 { v20.b }[4], [x23]\n"
      "st1 { v24.b }[4], [x22]\n"
      "b 133f\n"
      "131:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x10, #1, 132f\n"
      "str h8, [x11], #0x2\n"
      "str h12, [x25], #0x2\n"
      "str h16, [x24], #0x2\n"
      "str h20, [x23], #0x2\n"
      "str h24, [x22], #0x2\n"
      "tbz x10, #0, 133f\n"
      "st1 { v8.b }[2], [x11]\n"
      "st1 { v12.b }[2], [x25]\n"
      "st1 { v16.b }[2], [x24]\n"
      "st1 { v20.b }[2], [x23]\n"
      "st1 { v24.b }[2], [x22]\n"
      "b 133f\n"
      "132:"  // Height 5: Partial direct writeback: partial_1_0
      "str b8, [x11, #0x0]\n"
      "str b12, [x25, #0x0]\n"
      "str b16, [x24, #0x0]\n"
      "str b20, [x23, #0x0]\n"
      "str b24, [x22, #0x0]\n"
      "133:"  // Height 5: Partial direct writeback: Done
      "b 135f\n"
      "134:"  // Height 5: Full writeback
      "str q8, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q12, [x25, #0x0]\n"
      "str q16, [x24, #0x0]\n"
      "str q20, [x23, #0x0]\n"
      "str q24, [x22, #0x0]\n"
      "135:"  // Height 5: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 110b\n"
      "b 164f\n"
      "136:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x6\n"
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "mov x11, %x[output_ptr]\n"
      "madd %x[output_ptr], x21, x20, %x[output_ptr]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "137:"  // Height 6: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "138:"  // Height 6: setup done
      "mov x28, #0x0\n"
      "139:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 140f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 141f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 141f\n"
      "140:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "141:"  // Height 6: input setup done
      "cmp x27, #0x10\n"
      "blt 144f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q5, [x21, #0x0]\n"
      "ldr q6, [x9, #0x0]\n"
      "ldr q7, [x9, #0x10]\n"
      "blt 143f\n"
      "142:"  // Height 6: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0dc  // sdot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x9, #0x20]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0fd  // sdot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x9, #0x30]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0de  // sdot v30.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x9, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0ff  // sdot v31.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0d8  // sdot v24.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0dc  // sdot v28.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x9, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0f9  // sdot v25.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0fd  // sdot v29.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x9, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0da  // sdot v26.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0de  // sdot v30.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x9, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0fb  // sdot v27.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0ff  // sdot v31.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x9, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8d8  // sdot v24.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8dc  // sdot v28.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x9, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8f9  // sdot v25.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8fd  // sdot v29.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x9, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8da  // sdot v26.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8de  // sdot v30.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x9, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8fb  // sdot v27.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8ff  // sdot v31.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x9, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8d8  // sdot v24.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8dc  // sdot v28.4s, v6.16b, v5.4b[3]\n"
      "ldr q6, [x9, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8f9  // sdot v25.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8fd  // sdot v29.4s, v7.16b, v5.4b[3]\n"
      "ldr q7, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8da  // sdot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8de  // sdot v30.4s, v6.16b, v5.4b[3]\n"
      "ldr q6, [x9, #0x0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      "ldr q3, [x23, #0x0]\n"
      ".inst 0x4fa4e8fb  // sdot v27.4s, v7.16b, v4.4b[3]\n"
      "ldr q4, [x22, #0x0]\n"
      ".inst 0x4fa5e8ff  // sdot v31.4s, v7.16b, v5.4b[3]\n"
      "ldr q5, [x21, #0x0]\n"
      "ldr q7, [x9, #0x10]\n"
      "bge 142b\n"
      "143:"  // Height 6: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0dc  // sdot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x9, #0x20]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x21, x21, #0x10\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0fd  // sdot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x9, #0x30]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0de  // sdot v30.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x9, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0ff  // sdot v31.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0d8  // sdot v24.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0dc  // sdot v28.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x9, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0f9  // sdot v25.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0fd  // sdot v29.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x9, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0da  // sdot v26.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0de  // sdot v30.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x9, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0fb  // sdot v27.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0ff  // sdot v31.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x9, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8d8  // sdot v24.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8dc  // sdot v28.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x9, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8f9  // sdot v25.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8fd  // sdot v29.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x9, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8da  // sdot v26.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8de  // sdot v30.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x9, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8fb  // sdot v27.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8ff  // sdot v31.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x9, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8d8  // sdot v24.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8dc  // sdot v28.4s, v6.16b, v5.4b[3]\n"
      "ldr q6, [x9, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8f9  // sdot v25.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8fd  // sdot v29.4s, v7.16b, v5.4b[3]\n"
      "ldr q7, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8da  // sdot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8de  // sdot v30.4s, v6.16b, v5.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8fb  // sdot v27.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8ff  // sdot v31.4s, v7.16b, v5.4b[3]\n"
      "144:"  // Height 6: Multiply loop: Main loop skip
      "cbz x27, 149f\n"
      "cmp x27, #0x4\n"
      "blt 146f\n"
      "145:"  // Height 6: Multiply loop: Odd block loop
      "ldr s7, [x26], #0x4\n"
      "ldr s6, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "cmp x27, #0x4\n"
      "ldr s5, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "ldr s3, [x22], #0x4\n"
      "ldr s2, [x21], #0x4\n"
      "ldr q1, [x9, #0x0]\n"
      "ldr q0, [x9, #0x10]\n"
      ".inst 0x4f87e028  // sdot v8.4s, v1.16b, v7.4b[0]\n"
      ".inst 0x4f86e02c  // sdot v12.4s, v1.16b, v6.4b[0]\n"
      ".inst 0x4f85e030  // sdot v16.4s, v1.16b, v5.4b[0]\n"
      ".inst 0x4f84e034  // sdot v20.4s, v1.16b, v4.4b[0]\n"
      ".inst 0x4f83e038  // sdot v24.4s, v1.16b, v3.4b[0]\n"
      ".inst 0x4f82e03c  // sdot v28.4s, v1.16b, v2.4b[0]\n"
      "ldr q1, [x9, #0x20]\n"
      ".inst 0x4f87e009  // sdot v9.4s, v0.16b, v7.4b[0]\n"
      ".inst 0x4f86e00d  // sdot v13.4s, v0.16b, v6.4b[0]\n"
      ".inst 0x4f85e011  // sdot v17.4s, v0.16b, v5.4b[0]\n"
      ".inst 0x4f84e015  // sdot v21.4s, v0.16b, v4.4b[0]\n"
      ".inst 0x4f83e019  // sdot v25.4s, v0.16b, v3.4b[0]\n"
      ".inst 0x4f82e01d  // sdot v29.4s, v0.16b, v2.4b[0]\n"
      "ldr q0, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      ".inst 0x4f87e02a  // sdot v10.4s, v1.16b, v7.4b[0]\n"
      ".inst 0x4f86e02e  // sdot v14.4s, v1.16b, v6.4b[0]\n"
      ".inst 0x4f85e032  // sdot v18.4s, v1.16b, v5.4b[0]\n"
      ".inst 0x4f84e036  // sdot v22.4s, v1.16b, v4.4b[0]\n"
      ".inst 0x4f83e03a  // sdot v26.4s, v1.16b, v3.4b[0]\n"
      ".inst 0x4f82e03e  // sdot v30.4s, v1.16b, v2.4b[0]\n"
      ".inst 0x4f87e00b  // sdot v11.4s, v0.16b, v7.4b[0]\n"
      ".inst 0x4f86e00f  // sdot v15.4s, v0.16b, v6.4b[0]\n"
      ".inst 0x4f85e013  // sdot v19.4s, v0.16b, v5.4b[0]\n"
      ".inst 0x4f84e017  // sdot v23.4s, v0.16b, v4.4b[0]\n"
      ".inst 0x4f83e01b  // sdot v27.4s, v0.16b, v3.4b[0]\n"
      ".inst 0x4f82e01f  // sdot v31.4s, v0.16b, v2.4b[0]\n"
      "bge 145b\n"
      "146:"  // Height 6: Multiply loop: Skip odd blocks
      "cbz x27, 149f\n"
      "tbz x27, #1, 147f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr h4, [x22], #0x2\n"
      "ldr h5, [x21], #0x2\n"
      "tbz x27, #0, 148f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "ld1 { v3.b }[2], [x23]\n"
      "ld1 { v4.b }[2], [x22]\n"
      "ld1 { v5.b }[2], [x21]\n"
      "b 148f\n"
      "147:"  // Height 6: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "ldr b3, [x23, #0x0]\n"
      "ldr b4, [x22, #0x0]\n"
      "ldr b5, [x21, #0x0]\n"
      "148:"  // Height 6: Multiply loop: Ragged operand read: Done
      "ldr q7, [x9, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      ".inst 0x4f80e0e8  // sdot v8.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ec  // sdot v12.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f0  // sdot v16.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f4  // sdot v20.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f8  // sdot v24.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0fc  // sdot v28.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4f80e0c9  // sdot v9.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cd  // sdot v13.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d1  // sdot v17.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d5  // sdot v21.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0d9  // sdot v25.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0dd  // sdot v29.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      ".inst 0x4f80e0ea  // sdot v10.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ee  // sdot v14.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f2  // sdot v18.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f6  // sdot v22.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fa  // sdot v26.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0fe  // sdot v30.4s, v7.16b, v5.4b[0]\n"
      ".inst 0x4f80e0cb  // sdot v11.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cf  // sdot v15.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d3  // sdot v19.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d7  // sdot v23.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0db  // sdot v27.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0df  // sdot v31.4s, v6.16b, v5.4b[0]\n"
      "149:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 139b\n"
      "ldr q3, [x14, #0x0]\n"
      "ldr q2, [x14, #0x10]\n"
      "add v8.4s, v8.4s, v3.4s\n"
      "add v9.4s, v9.4s, v2.4s\n"
      "ldr q1, [x14, #0x20]\n"
      "ldr q0, [x14, #0x30]\n"
      "add v10.4s, v10.4s, v1.4s\n"
      "add v11.4s, v11.4s, v0.4s\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x11, x20\n"
      "add x24, x25, x20\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "add x23, x24, x20\n"
      "add x22, x23, x20\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add x21, x22, x20\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "add v12.4s, v12.4s, v3.4s\n"
      "prfm pstl1keep, [x21, #0x0]\n"
      "add v13.4s, v13.4s, v2.4s\n"
      "add v14.4s, v14.4s, v1.4s\n"
      "add x14, x14, #0x40\n"
      "add v15.4s, v15.4s, v0.4s\n"
      "add v16.4s, v16.4s, v3.4s\n"
      "add v17.4s, v17.4s, v2.4s\n"
      "add v18.4s, v18.4s, v1.4s\n"
      "add v19.4s, v19.4s, v0.4s\n"
      "add v20.4s, v20.4s, v3.4s\n"
      "add v21.4s, v21.4s, v2.4s\n"
      "add v22.4s, v22.4s, v1.4s\n"
      "add v23.4s, v23.4s, v0.4s\n"
      "add v24.4s, v24.4s, v3.4s\n"
      "add v25.4s, v25.4s, v2.4s\n"
      "add v26.4s, v26.4s, v1.4s\n"
      "add v27.4s, v27.4s, v0.4s\n"
      "add v28.4s, v28.4s, v3.4s\n"
      "add v29.4s, v29.4s, v2.4s\n"
      "add v30.4s, v30.4s, v1.4s\n"
      "add v31.4s, v31.4s, v0.4s\n"
      "tbz %x[flags], #4, 150f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 151f\n"
      "150:"  // Height 6: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "151:"  // Height 6: parameters loaded
      "sqrdmulh v8.4s, v8.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "sqrdmulh v12.4s, v12.4s, v4.4s\n"
      "sqrdmulh v13.4s, v13.4s, v5.4s\n"
      "sqrdmulh v14.4s, v14.4s, v6.4s\n"
      "sqrdmulh v15.4s, v15.4s, v7.4s\n"
      "sqrdmulh v16.4s, v16.4s, v4.4s\n"
      "sqrdmulh v17.4s, v17.4s, v5.4s\n"
      "sqrdmulh v18.4s, v18.4s, v6.4s\n"
      "sqrdmulh v19.4s, v19.4s, v7.4s\n"
      "sqrdmulh v20.4s, v20.4s, v4.4s\n"
      "sqrdmulh v21.4s, v21.4s, v5.4s\n"
      "sqrdmulh v22.4s, v22.4s, v6.4s\n"
      "sqrdmulh v23.4s, v23.4s, v7.4s\n"
      "sqrdmulh v24.4s, v24.4s, v4.4s\n"
      "sqrdmulh v25.4s, v25.4s, v5.4s\n"
      "sqrdmulh v26.4s, v26.4s, v6.4s\n"
      "sqrdmulh v27.4s, v27.4s, v7.4s\n"
      "sqrdmulh v28.4s, v28.4s, v4.4s\n"
      "sqrdmulh v29.4s, v29.4s, v5.4s\n"
      "sqrdmulh v30.4s, v30.4s, v6.4s\n"
      "sqrdmulh v31.4s, v31.4s, v7.4s\n"
      "tbz %x[flags], #5, 152f\n"
      "and v7.16b, v8.16b, v0.16b\n"
      "and v6.16b, v9.16b, v1.16b\n"
      "and v5.16b, v10.16b, v2.16b\n"
      "and v4.16b, v11.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v8.4s, v8.4s, v7.4s\n"
      "sqadd v9.4s, v9.4s, v6.4s\n"
      "sqadd v10.4s, v10.4s, v5.4s\n"
      "sqadd v11.4s, v11.4s, v4.4s\n"
      "and v7.16b, v12.16b, v0.16b\n"
      "and v6.16b, v13.16b, v1.16b\n"
      "and v5.16b, v14.16b, v2.16b\n"
      "and v4.16b, v15.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v12.4s, v12.4s, v7.4s\n"
      "sqadd v13.4s, v13.4s, v6.4s\n"
      "sqadd v14.4s, v14.4s, v5.4s\n"
      "sqadd v15.4s, v15.4s, v4.4s\n"
      "and v7.16b, v16.16b, v0.16b\n"
      "and v6.16b, v17.16b, v1.16b\n"
      "and v5.16b, v18.16b, v2.16b\n"
      "and v4.16b, v19.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v7.4s\n"
      "sqadd v17.4s, v17.4s, v6.4s\n"
      "sqadd v18.4s, v18.4s, v5.4s\n"
      "sqadd v19.4s, v19.4s, v4.4s\n"
      "and v7.16b, v20.16b, v0.16b\n"
      "and v6.16b, v21.16b, v1.16b\n"
      "and v5.16b, v22.16b, v2.16b\n"
      "and v4.16b, v23.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v20.4s, v20.4s, v7.4s\n"
      "sqadd v21.4s, v21.4s, v6.4s\n"
      "sqadd v22.4s, v22.4s, v5.4s\n"
      "sqadd v23.4s, v23.4s, v4.4s\n"
      "and v7.16b, v24.16b, v0.16b\n"
      "and v6.16b, v25.16b, v1.16b\n"
      "and v5.16b, v26.16b, v2.16b\n"
      "and v4.16b, v27.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v24.4s, v24.4s, v7.4s\n"
      "sqadd v25.4s, v25.4s, v6.4s\n"
      "sqadd v26.4s, v26.4s, v5.4s\n"
      "sqadd v27.4s, v27.4s, v4.4s\n"
      "and v7.16b, v28.16b, v0.16b\n"
      "and v6.16b, v29.16b, v1.16b\n"
      "and v5.16b, v30.16b, v2.16b\n"
      "and v4.16b, v31.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v28.4s, v28.4s, v7.4s\n"
      "sqadd v29.4s, v29.4s, v6.4s\n"
      "sqadd v30.4s, v30.4s, v5.4s\n"
      "sqadd v31.4s, v31.4s, v4.4s\n"
      "152:"  // Height 6: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v6.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v5.4s }, [x20]\n"
      "srshl v12.4s, v12.4s, v0.4s\n"
      "srshl v13.4s, v13.4s, v1.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v4.4s }, [x20]\n"
      "srshl v14.4s, v14.4s, v2.4s\n"
      "srshl v15.4s, v15.4s, v3.4s\n"
      "cmp x10, #0x10\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v1.4s\n"
      "srshl v22.4s, v22.4s, v2.4s\n"
      "srshl v23.4s, v23.4s, v3.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v1.4s\n"
      "srshl v26.4s, v26.4s, v2.4s\n"
      "srshl v27.4s, v27.4s, v3.4s\n"
      "srshl v28.4s, v28.4s, v0.4s\n"
      "srshl v29.4s, v29.4s, v1.4s\n"
      "srshl v30.4s, v30.4s, v2.4s\n"
      "srshl v31.4s, v31.4s, v3.4s\n"
      "add v8.4s, v8.4s, v6.4s\n"
      "add v9.4s, v9.4s, v6.4s\n"
      "add v10.4s, v10.4s, v6.4s\n"
      "add v11.4s, v11.4s, v6.4s\n"
      "add v12.4s, v12.4s, v6.4s\n"
      "add v13.4s, v13.4s, v6.4s\n"
      "add v14.4s, v14.4s, v6.4s\n"
      "add v15.4s, v15.4s, v6.4s\n"
      "add v16.4s, v16.4s, v6.4s\n"
      "add v17.4s, v17.4s, v6.4s\n"
      "add v18.4s, v18.4s, v6.4s\n"
      "add v19.4s, v19.4s, v6.4s\n"
      "add v20.4s, v20.4s, v6.4s\n"
      "add v21.4s, v21.4s, v6.4s\n"
      "add v22.4s, v22.4s, v6.4s\n"
      "add v23.4s, v23.4s, v6.4s\n"
      "add v24.4s, v24.4s, v6.4s\n"
      "add v25.4s, v25.4s, v6.4s\n"
      "add v26.4s, v26.4s, v6.4s\n"
      "add v27.4s, v27.4s, v6.4s\n"
      "add v28.4s, v28.4s, v6.4s\n"
      "add v29.4s, v29.4s, v6.4s\n"
      "add v30.4s, v30.4s, v6.4s\n"
      "add v31.4s, v31.4s, v6.4s\n"
      "smin v8.4s, v8.4s, v5.4s\n"
      "smin v9.4s, v9.4s, v5.4s\n"
      "smin v10.4s, v10.4s, v5.4s\n"
      "smin v11.4s, v11.4s, v5.4s\n"
      "smin v12.4s, v12.4s, v5.4s\n"
      "smin v13.4s, v13.4s, v5.4s\n"
      "smin v14.4s, v14.4s, v5.4s\n"
      "smin v15.4s, v15.4s, v5.4s\n"
      "smin v16.4s, v16.4s, v5.4s\n"
      "smin v17.4s, v17.4s, v5.4s\n"
      "smin v18.4s, v18.4s, v5.4s\n"
      "smin v19.4s, v19.4s, v5.4s\n"
      "smin v20.4s, v20.4s, v5.4s\n"
      "smin v21.4s, v21.4s, v5.4s\n"
      "smin v22.4s, v22.4s, v5.4s\n"
      "smin v23.4s, v23.4s, v5.4s\n"
      "smin v24.4s, v24.4s, v5.4s\n"
      "smin v25.4s, v25.4s, v5.4s\n"
      "smin v26.4s, v26.4s, v5.4s\n"
      "smin v27.4s, v27.4s, v5.4s\n"
      "smin v28.4s, v28.4s, v5.4s\n"
      "smin v29.4s, v29.4s, v5.4s\n"
      "smin v30.4s, v30.4s, v5.4s\n"
      "smin v31.4s, v31.4s, v5.4s\n"
      "smax v8.4s, v8.4s, v4.4s\n"
      "smax v9.4s, v9.4s, v4.4s\n"
      "smax v10.4s, v10.4s, v4.4s\n"
      "smax v11.4s, v11.4s, v4.4s\n"
      "smax v12.4s, v12.4s, v4.4s\n"
      "smax v13.4s, v13.4s, v4.4s\n"
      "smax v14.4s, v14.4s, v4.4s\n"
      "smax v15.4s, v15.4s, v4.4s\n"
      "smax v16.4s, v16.4s, v4.4s\n"
      "smax v17.4s, v17.4s, v4.4s\n"
      "smax v18.4s, v18.4s, v4.4s\n"
      "smax v19.4s, v19.4s, v4.4s\n"
      "smax v20.4s, v20.4s, v4.4s\n"
      "smax v21.4s, v21.4s, v4.4s\n"
      "smax v22.4s, v22.4s, v4.4s\n"
      "smax v23.4s, v23.4s, v4.4s\n"
      "smax v24.4s, v24.4s, v4.4s\n"
      "smax v25.4s, v25.4s, v4.4s\n"
      "smax v26.4s, v26.4s, v4.4s\n"
      "smax v27.4s, v27.4s, v4.4s\n"
      "smax v28.4s, v28.4s, v4.4s\n"
      "smax v29.4s, v29.4s, v4.4s\n"
      "smax v30.4s, v30.4s, v4.4s\n"
      "smax v31.4s, v31.4s, v4.4s\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v2.8h, v10.8h, v11.8h\n"
      "uzp1 v12.8h, v12.8h, v13.8h\n"
      "uzp1 v1.8h, v14.8h, v15.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v0.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "uzp1 v19.8h, v22.8h, v23.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v18.8h, v26.8h, v27.8h\n"
      "uzp1 v28.8h, v28.8h, v29.8h\n"
      "uzp1 v17.8h, v30.8h, v31.8h\n"
      "uzp1 v8.16b, v8.16b, v2.16b\n"
      "uzp1 v12.16b, v12.16b, v1.16b\n"
      "uzp1 v16.16b, v16.16b, v0.16b\n"
      "uzp1 v20.16b, v20.16b, v19.16b\n"
      "uzp1 v24.16b, v24.16b, v18.16b\n"
      "uzp1 v28.16b, v28.16b, v17.16b\n"
      "bge 161f\n"
      "tbz x10, #3, 156f\n"
      "str d8, [x11], #0x8\n"
      "str d12, [x25], #0x8\n"
      "str d16, [x24], #0x8\n"
      "str d20, [x23], #0x8\n"
      "str d24, [x22], #0x8\n"
      "str d28, [x21], #0x8\n"
      "tbz x10, #2, 154f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "st1 { v12.s }[2], [x25], #0x4\n"
      "st1 { v16.s }[2], [x24], #0x4\n"
      "st1 { v20.s }[2], [x23], #0x4\n"
      "st1 { v24.s }[2], [x22], #0x4\n"
      "st1 { v28.s }[2], [x21], #0x4\n"
      "tbz x10, #1, 153f\n"
      "st1 { v8.h }[6], [x11], #0x2\n"
      "st1 { v12.h }[6], [x25], #0x2\n"
      "st1 { v16.h }[6], [x24], #0x2\n"
      "st1 { v20.h }[6], [x23], #0x2\n"
      "st1 { v24.h }[6], [x22], #0x2\n"
      "st1 { v28.h }[6], [x21], #0x2\n"
      "tbz x10, #0, 160f\n"
      "st1 { v8.b }[14], [x11]\n"
      "st1 { v12.b }[14], [x25]\n"
      "st1 { v16.b }[14], [x24]\n"
      "st1 { v20.b }[14], [x23]\n"
      "st1 { v24.b }[14], [x22]\n"
      "st1 { v28.b }[14], [x21]\n"
      "b 160f\n"
      "153:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x10, #0, 160f\n"
      "st1 { v8.b }[12], [x11]\n"
      "st1 { v12.b }[12], [x25]\n"
      "st1 { v16.b }[12], [x24]\n"
      "st1 { v20.b }[12], [x23]\n"
      "st1 { v24.b }[12], [x22]\n"
      "st1 { v28.b }[12], [x21]\n"
      "b 160f\n"
      "154:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x10, #1, 155f\n"
      "st1 { v8.h }[4], [x11], #0x2\n"
      "st1 { v12.h }[4], [x25], #0x2\n"
      "st1 { v16.h }[4], [x24], #0x2\n"
      "st1 { v20.h }[4], [x23], #0x2\n"
      "st1 { v24.h }[4], [x22], #0x2\n"
      "st1 { v28.h }[4], [x21], #0x2\n"
      "tbz x10, #0, 160f\n"
      "st1 { v8.b }[10], [x11]\n"
      "st1 { v12.b }[10], [x25]\n"
      "st1 { v16.b }[10], [x24]\n"
      "st1 { v20.b }[10], [x23]\n"
      "st1 { v24.b }[10], [x22]\n"
      "st1 { v28.b }[10], [x21]\n"
      "b 160f\n"
      "155:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x10, #0, 160f\n"
      "st1 { v8.b }[8], [x11]\n"
      "st1 { v12.b }[8], [x25]\n"
      "st1 { v16.b }[8], [x24]\n"
      "st1 { v20.b }[8], [x23]\n"
      "st1 { v24.b }[8], [x22]\n"
      "st1 { v28.b }[8], [x21]\n"
      "b 160f\n"
      "156:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x10, #2, 158f\n"
      "str s8, [x11], #0x4\n"
      "str s12, [x25], #0x4\n"
      "str s16, [x24], #0x4\n"
      "str s20, [x23], #0x4\n"
      "str s24, [x22], #0x4\n"
      "str s28, [x21], #0x4\n"
      "tbz x10, #1, 157f\n"
      "st1 { v8.h }[2], [x11], #0x2\n"
      "st1 { v12.h }[2], [x25], #0x2\n"
      "st1 { v16.h }[2], [x24], #0x2\n"
      "st1 { v20.h }[2], [x23], #0x2\n"
      "st1 { v24.h }[2], [x22], #0x2\n"
      "st1 { v28.h }[2], [x21], #0x2\n"
      "tbz x10, #0, 160f\n"
      "st1 { v8.b }[6], [x11]\n"
      "st1 { v12.b }[6], [x25]\n"
      "st1 { v16.b }[6], [x24]\n"
      "st1 { v20.b }[6], [x23]\n"
      "st1 { v24.b }[6], [x22]\n"
      "st1 { v28.b }[6], [x21]\n"
      "b 160f\n"
      "157:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x10, #0, 160f\n"
      "st1 { v8.b }[4], [x11]\n"
      "st1 { v12.b }[4], [x25]\n"
      "st1 { v16.b }[4], [x24]\n"
      "st1 { v20.b }[4], [x23]\n"
      "st1 { v24.b }[4], [x22]\n"
      "st1 { v28.b }[4], [x21]\n"
      "b 160f\n"
      "158:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x10, #1, 159f\n"
      "str h8, [x11], #0x2\n"
      "str h12, [x25], #0x2\n"
      "str h16, [x24], #0x2\n"
      "str h20, [x23], #0x2\n"
      "str h24, [x22], #0x2\n"
      "str h28, [x21], #0x2\n"
      "tbz x10, #0, 160f\n"
      "st1 { v8.b }[2], [x11]\n"
      "st1 { v12.b }[2], [x25]\n"
      "st1 { v16.b }[2], [x24]\n"
      "st1 { v20.b }[2], [x23]\n"
      "st1 { v24.b }[2], [x22]\n"
      "st1 { v28.b }[2], [x21]\n"
      "b 160f\n"
      "159:"  // Height 6: Partial direct writeback: partial_1_0
      "str b8, [x11, #0x0]\n"
      "str b12, [x25, #0x0]\n"
      "str b16, [x24, #0x0]\n"
      "str b20, [x23, #0x0]\n"
      "str b24, [x22, #0x0]\n"
      "str b28, [x21, #0x0]\n"
      "160:"  // Height 6: Partial direct writeback: Done
      "b 162f\n"
      "161:"  // Height 6: Full writeback
      "str q8, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q12, [x25, #0x0]\n"
      "str q16, [x24, #0x0]\n"
      "str q20, [x23, #0x0]\n"
      "str q24, [x22, #0x0]\n"
      "str q28, [x21, #0x0]\n"
      "162:"  // Height 6: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 137b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 164f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 163f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "163:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "164:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr), [output_ptr] "+&r" (output_ptr)
      : [args_ptr] "r" (&ka), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [flags] "r" (flags), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_multiplier_ptr] "I" (offsetof(KernelArgs, multiplier_ptr)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_shift_ptr] "I" (offsetof(KernelArgs, shift_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x13", "x14", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
