// Copyright (c) 2013-2020, SIB - Swiss Institute of Bioinformatics and
//                          Biozentrum - University of Basel
// 
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// 
//   http://www.apache.org/licenses/LICENSE-2.0
// 
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.


#include <promod3/loop/sidechain_atom_rule_lookup.hh>
#include <promod3/loop/amino_acid_atoms.hh>
#include <ost/conop/amino_acids.hh>
#include <promod3/core/message.hh>

namespace promod3 { namespace loop{


ChiDefinition SidechainAtomRuleLookup::GetChiDefinition(ost::conop::AminoAcid aa, 
                                                     uint chi_idx) const{
  if(chi_idx >= chi_definitions_[aa].size()){
    String err = "Specified sidechain dihedral does not exist for ";
    err += (ost::conop::AminoAcidToResidueName(aa) + "!");
    throw promod3::Error(err);
  }
  return chi_definitions_[aa][chi_idx];
}

void SidechainAtomRuleLookup::AddRule(ost::conop::AminoAcid aa, 
                                      uint sidechain_atom_idx,
                                      uint anch_1, uint anch_2, uint anch_3, 
                                      Real bond_length, Real angle, 
                                      uint dihedral_idx, 
                                      Real base_dihedral){
  SidechainAtomRule rule;
  rule.sidechain_atom_idx = sidechain_atom_idx;
  rule.anchor_idx[0] = anch_1;
  rule.anchor_idx[1] = anch_2;
  rule.anchor_idx[2] = anch_3;
  rule.bond_length = bond_length;
  rule.angle = angle;
  rule.dihedral_idx = dihedral_idx;
  rule.base_dihedral = base_dihedral;
  rules_[aa].push_back(rule);
}

void SidechainAtomRuleLookup::AddChiDefinition(ost::conop::AminoAcid aa,
                                           uint idx_one, uint idx_two, 
                                           uint idx_three, uint idx_four){
  ChiDefinition chi_definition;
  chi_definition.idx_one = idx_one;
  chi_definition.idx_two = idx_two;
  chi_definition.idx_three = idx_three;
  chi_definition.idx_four = idx_four;
  chi_definitions_[aa].push_back(chi_definition);
}

SidechainAtomRuleLookup::SidechainAtomRuleLookup(){
    
  // ARG
  AddRule(ost::conop::ARG, 5, 0, 1, 4,
          Real(1.5475), Real(2.02370926769), 0, Real(0.0));
  AddRule(ost::conop::ARG, 6, 1, 4, 5,
          Real(1.5384), Real(1.9898498802), 1, Real(0.0));
  AddRule(ost::conop::ARG, 7, 4, 5, 6,
          Real(1.5034), Real(1.86907309596), 2, Real(0.0));
  AddRule(ost::conop::ARG, 8, 5, 6, 7,
          Real(1.3401), Real(2.14762764458), 3, Real(0.0));
  AddRule(ost::conop::ARG, 9, 6, 7, 8,
          Real(1.3311), Real(2.0605357149), 4, Real(3.14159265359));
  AddRule(ost::conop::ARG, 10, 6, 7, 8,
          Real(1.3292), Real(2.13174514839), 4, Real(0.0));

  // ASN
  AddRule(ost::conop::ASN, 5, 0, 1, 4,
          Real(1.5319), Real(1.99491133503), 0, Real(0.0));
  AddRule(ost::conop::ASN, 6, 1, 4, 5,
          Real(1.2323), Real(2.13907553124), 1, Real(0.0));
  AddRule(ost::conop::ASN, 7, 1, 4, 5,
          Real(1.3521), Real(2.02719992619), 1, Real(3.14159265359));

  // ASP
  AddRule(ost::conop::ASP, 5, 0, 1, 4,
          Real(1.5218), Real(1.96524073775), 0, Real(0.0));
  AddRule(ost::conop::ASP, 6, 1, 4, 5,
          Real(1.2565), Real(2.05931398443), 1, Real(0.0));
  AddRule(ost::conop::ASP, 7, 1, 4, 5,
          Real(1.2541), Real(2.0542525296), 1, Real(3.14159265359));

  // GLN
  AddRule(ost::conop::GLN, 5, 0, 1, 4,
          Real(1.5534), Real(2.0162043519), 0, Real(0.0));
  AddRule(ost::conop::GLN, 6, 1, 4, 5,
          Real(1.532), Real(1.96349540849), 1, Real(0.0));
  AddRule(ost::conop::GLN, 7, 4, 5, 6,
          Real(1.2294), Real(2.12092410702), 2, Real(0.0));
  AddRule(ost::conop::GLN, 8, 4, 5, 6,
          Real(1.353), Real(2.03924269803), 2, Real(3.14159265359));

  // GLU
  AddRule(ost::conop::GLU, 5, 0, 1, 4,
          Real(1.5557), Real(2.01917141163), 0, Real(0.0));
  AddRule(ost::conop::GLU, 6, 1, 4, 5,
          Real(1.5307), Real(2.01986954333), 1, Real(0.0));
  AddRule(ost::conop::GLU, 7, 4, 5, 6,
          Real(1.259), Real(2.00695410687), 2, Real(0.0));
  AddRule(ost::conop::GLU, 8, 4, 5, 6,
          Real(1.2532), Real(2.09579136579), 2, Real(3.14159265359));

  // LYS
  AddRule(ost::conop::LYS, 5, 0, 1, 4,
          Real(1.5435), Real(2.02039314211), 0, Real(0.0));
  AddRule(ost::conop::LYS, 6, 1, 4, 5,
          Real(1.5397), Real(1.97710897666), 1, Real(0.0));
  AddRule(ost::conop::LYS, 7, 4, 5, 6,
          Real(1.535), Real(1.96052834877), 2, Real(0.0));
  AddRule(ost::conop::LYS, 8, 5, 6, 7,
          Real(1.4604), Real(1.92789069175), 3, Real(0.0));

  // SER
  AddRule(ost::conop::SER, 5, 0, 1, 4,
          Real(1.4341), Real(1.96262274387), 0, Real(0.0));

  // CYS
  AddRule(ost::conop::CYS, 5, 0, 1, 4,
          Real(1.8359), Real(1.98740641925), 0, Real(0.0));

  // MET
  AddRule(ost::conop::MET, 5, 0, 1, 4,
          Real(1.546), Real(2.02318566891), 0, Real(0.0));
  AddRule(ost::conop::MET, 6, 1, 4, 5,
          Real(1.8219), Real(1.9247490991), 1, Real(0.0));
  AddRule(ost::conop::MET, 7, 4, 5, 6,
          Real(1.8206), Real(1.72682876192), 2, Real(0.0));

  // TRP
  AddRule(ost::conop::TRP, 5, 0, 1, 4,
          Real(1.5233), Real(2.00957210075), 0, Real(0.0));
  AddRule(ost::conop::TRP, 6, 1, 4, 5,
          Real(1.3679), Real(2.25461632773), 1, Real(0.0));
  AddRule(ost::conop::TRP, 7, 1, 4, 5,
          Real(1.4407), Real(2.16333560785), 1, Real(3.14159265359));
  AddRule(ost::conop::TRP, 8, 6, 5, 7,
          Real(1.4126), Real(1.86139364725), 4, Real(0.0));
  AddRule(ost::conop::TRP, 9, 5, 7, 8,
          Real(1.3746), Real(1.88268666413), 4, Real(0.0));
  AddRule(ost::conop::TRP, 10, 6, 5, 7,
          Real(1.4011), Real(2.31325939059), 4, Real(3.14159265359));
  AddRule(ost::conop::TRP, 11, 8, 7, 10,
          Real(1.4017), Real(2.06228104416), 4, Real(0.0));
  AddRule(ost::conop::TRP, 12, 7, 10, 11,
          Real(1.4019), Real(2.11132479614), 4, Real(0.0));
  AddRule(ost::conop::TRP, 13, 10, 11, 12,
          Real(1.403), Real(2.10957946689), 4, Real(0.0));

  // TYR
  AddRule(ost::conop::TYR, 5, 0, 1, 4,
          Real(1.5113), Real(1.9711748572), 0, Real(0.0));
  AddRule(ost::conop::TYR, 6, 1, 4, 5,
          Real(1.4064), Real(2.10294721573), 1, Real(0.0));
  AddRule(ost::conop::TYR, 7, 1, 4, 5,
          Real(1.4068), Real(2.10242361695), 1, Real(3.14159265359));
  AddRule(ost::conop::TYR, 8, 7, 5, 6,
          Real(1.4026), Real(2.1013764194), 4, Real(0.0));
  AddRule(ost::conop::TYR, 9, 6, 5, 7,
          Real(1.4022), Real(2.1041689462), 4, Real(0.0));
  AddRule(ost::conop::TYR, 10, 5, 6, 8,
          Real(1.3978), Real(2.09596589872), 4, Real(0.0));
  AddRule(ost::conop::TYR, 11, 7, 9, 10,
          Real(1.4063), Real(2.09875842552), 4, Real(3.14159265359));

  // THR
  AddRule(ost::conop::THR, 5, 0, 1, 4,
          Real(1.4252), Real(1.95756128904), 0, Real(0.0));
  AddRule(ost::conop::THR, 6, 5, 1, 4,
          Real(1.5324), Real(2.02301113599), 4, Real(-2.1665));

  // VAL
  AddRule(ost::conop::VAL, 5, 0, 1, 4,
          Real(1.5441), Real(1.9891517485), 0, Real(0.0));
  AddRule(ost::conop::VAL, 6, 5, 1, 4,
          Real(1.5414), Real(1.95773582196), 4, Real(2.164));

  // ILE
  AddRule(ost::conop::ILE, 5, 0, 1, 4,
          Real(1.5498), Real(1.98321762904), 0, Real(0.0));
  AddRule(ost::conop::ILE, 6, 5, 1, 4,
          Real(1.5452), Real(1.9884536168), 4, Real(-2.2696));
  AddRule(ost::conop::ILE, 7, 1, 4, 5,
          Real(1.5381), Real(1.9912461436), 1, Real(0.0));

  // LEU
  AddRule(ost::conop::LEU, 5, 0, 1, 4,
          Real(1.5472), Real(2.05006373939), 0, Real(0.0));
  AddRule(ost::conop::LEU, 6, 1, 4, 5,
          Real(1.5361), Real(1.9282397576), 1, Real(0));
  AddRule(ost::conop::LEU, 7, 6, 4, 5,
          Real(1.536), Real(1.96471713897), 4, Real(2.0944));

  // PRO
  AddRule(ost::conop::PRO, 5, 0, 1, 4,
          Real(1.5322), Real(1.82194920616), 0, Real(0.0));
  AddRule(ost::conop::PRO, 6, 1, 4, 5,
          Real(1.5317), Real(1.80135432098), 1, Real(0.0));

  // HIS
  AddRule(ost::conop::HIS, 5, 0, 1, 4,
          Real(1.5109), Real(2.04098802728), 0, Real(0.0));
  AddRule(ost::conop::HIS, 6, 1, 4, 5,
          Real(1.3859), Real(2.09736216212), 1, Real(0.0));
  AddRule(ost::conop::HIS, 7, 1, 4, 5,
          Real(1.3596), Real(2.26386657276), 1, Real(3.14159265359));
  AddRule(ost::conop::HIS, 8, 7, 5, 6,
          Real(1.317), Real(1.8360863731), 4, Real(0.0));
  AddRule(ost::conop::HIS, 9, 6, 5, 7,
          Real(1.3782), Real(1.84655834861), 4, Real(0.0));

  // PHE
  AddRule(ost::conop::PHE, 5, 0, 1, 4,
          Real(1.5109), Real(1.96803326455), 0, Real(0.0));
  AddRule(ost::conop::PHE, 6, 1, 4, 5,
          Real(1.4059), Real(2.099980156), 1, Real(0.0));
  AddRule(ost::conop::PHE, 7, 1, 4, 5,
          Real(1.4062), Real(2.10765960471), 1, Real(3.14159265359));
  AddRule(ost::conop::PHE, 8, 7, 5, 6,
          Real(1.4006), Real(2.10539067668), 4, Real(0.0));
  AddRule(ost::conop::PHE, 9, 6, 5, 7,
          Real(1.4002), Real(2.10521614376), 4, Real(0.0));
  AddRule(ost::conop::PHE, 10, 5, 6, 8,
          Real(1.4004), Real(2.09317337192), 4, Real(0.0));

  // ARG
  AddChiDefinition(ost::conop::ARG, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::ARG, 1, 4, 5, 6);
  AddChiDefinition(ost::conop::ARG, 4, 5, 6, 7);
  AddChiDefinition(ost::conop::ARG, 5, 6, 7, 8);

  // ASN
  AddChiDefinition(ost::conop::ASN, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::ASN, 1, 4, 5, 6);

  // ASP
  AddChiDefinition(ost::conop::ASP, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::ASP, 1, 4, 5, 6);

  // GLN
  AddChiDefinition(ost::conop::GLN, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::GLN, 1, 4, 5, 6);
  AddChiDefinition(ost::conop::GLN, 4, 5, 6, 7);

  // GLU
  AddChiDefinition(ost::conop::GLU, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::GLU, 1, 4, 5, 6);
  AddChiDefinition(ost::conop::GLU, 4, 5, 6, 7);

  // LYS
  AddChiDefinition(ost::conop::LYS, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::LYS, 1, 4, 5, 6);
  AddChiDefinition(ost::conop::LYS, 4, 5, 6, 7);
  AddChiDefinition(ost::conop::LYS, 5, 6, 7, 8);

  // SER
  AddChiDefinition(ost::conop::SER, 0, 1, 4, 5);

  // CYS
  AddChiDefinition(ost::conop::CYS, 0, 1, 4, 5);

  // MET
  AddChiDefinition(ost::conop::MET, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::MET, 1, 4, 5, 6);
  AddChiDefinition(ost::conop::MET, 4, 5, 6, 7);

  // TRP
  AddChiDefinition(ost::conop::TRP, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::TRP, 1, 4, 5, 6);

  // TYR
  AddChiDefinition(ost::conop::TYR, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::TYR, 1, 4, 5, 6);

  // THR
  AddChiDefinition(ost::conop::THR, 0, 1, 4, 5);

  // VAL
  AddChiDefinition(ost::conop::VAL, 0, 1, 4, 5);

  // ILE
  AddChiDefinition(ost::conop::ILE, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::ILE, 1, 4, 5, 7);

  // LEU
  AddChiDefinition(ost::conop::LEU, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::LEU, 1, 4, 5, 6);

  // PRO
  AddChiDefinition(ost::conop::PRO, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::PRO, 1, 4, 5, 6);

  // HIS
  AddChiDefinition(ost::conop::HIS, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::HIS, 1, 4, 5, 6);

  // PHE
  AddChiDefinition(ost::conop::PHE, 0, 1, 4, 5);
  AddChiDefinition(ost::conop::PHE, 1, 4, 5, 6);


}


}}
