/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4; fill-column: 100 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include <iostream>

#include "rustoptions.hxx"

bool RustOptions::initOptions(int argc, char* argv[], bool)
{
    if (argc < 2)
    {
        std::cerr << prepareHelp();
        return false;
    }

    for (int i = 1; i < argc; i++)
    {
        OString argument = argv[i];

        if (argument == "-h"_ostr || argument == "--help"_ostr)
        {
            std::cout << prepareHelp();
            return false;
        }
        else if (argument == "-v"_ostr || argument == "--verbose"_ostr)
        {
            m_options["--verbose"_ostr] = ""_ostr;
        }
        else if (argument == "-n"_ostr || argument == "--dry-run"_ostr)
        {
            m_options["--dry-run"_ostr] = ""_ostr;
            m_options["--verbose"_ostr] = "--dry-run"_ostr; // dry run implies verbose
        }
        else if (argument == "-T"_ostr || argument == "--types"_ostr)
        {
            if (i + 1 < argc)
            {
                if (m_options.count("--types"_ostr) == 0)
                {
                    m_options["--types"_ostr] = argv[++i];
                }
                else
                {
                    // Allow multiple -T options by joining with semicolon
                    m_options["--types"_ostr] += ";"_ostr + argv[++i];
                }
            }
            else
            {
                throw IllegalArgument("-T/--types must be followed by type name or wildcard"_ostr);
            }
        }
        else if (argument == "-X"_ostr || argument == "--extra-types"_ostr)
        {
            if (i + 1 < argc)
            {
                m_extra_input_files.emplace_back(argv[++i]);
            }
            else
            {
                throw IllegalArgument("-X/--extra-types must be followed by .rdb file"_ostr);
            }
        }
        else if (argument == "-Ocpp"_ostr)
        {
            if (i + 1 < argc)
            {
                m_options["--cpp-output-dir"_ostr] = argv[++i];
            }
            else
            {
                throw IllegalArgument("-Ocpp must be followed by directory"_ostr);
            }
        }
        else if (argument == "-Orust"_ostr)
        {
            if (i + 1 < argc)
            {
                m_options["--rust-output-dir"_ostr] = argv[++i];
            }
            else
            {
                throw IllegalArgument("-Orust must be followed by directory"_ostr);
            }
        }
        else if (argument == "-O"_ostr || argument == "--output-dir"_ostr)
        {
            if (i + 1 < argc)
            {
                // Legacy support: set both cpp and rust output to same directory
                m_options["--cpp-output-dir"_ostr]
                    = OString::Concat(argv[++i]) + "/generated/cpp_rustmaker";
                m_options["--rust-output-dir"_ostr] = OString::Concat(argv[i]) + "/generated";
            }
            else
            {
                throw IllegalArgument("-O/--output-dir must be followed by directory"_ostr);
            }
        }
        else
        {
            // Any non-option argument is treated as input .rdb file
            m_inputFiles.emplace_back(argument);
        }
    }

    // Validate required arguments
    if (m_inputFiles.empty())
    {
        throw IllegalArgument("at least one .rdb file must be provided"_ostr);
    }

    if (m_options.count("--cpp-output-dir"_ostr) == 0
        || m_options.count("--rust-output-dir"_ostr) == 0)
    {
        throw IllegalArgument(
            "Both -Ocpp and -Orust must be provided (or use -O for legacy mode)"_ostr);
    }

    return true;
}

OString RustOptions::prepareHelp()
{
    return R"(

About:
    rustmaker is a tool for generating Rust files from a type library generated by the UNOIDL compiler unoidl-write.
    The generated code files require the implemented Rust types from rust_uno.

Usage:
    rustmaker [-v|--verbose] [-n|--dry-run]
        [-T|--types <type name or wildcard>]
        [-X|--extra-types <.rdb file>]
        -Ocpp <cpp output directory> -Orust <rust output directory>
        <rdb file(s)>

    OR (legacy mode):
    rustmaker [-v|--verbose] [-n|--dry-run]
        [-T|--types <type name or wildcard>]
        [-X|--extra-types <.rdb file>]
        -O|--output-dir <output directory>
        <rdb file(s)>

Options:
    -h, --help
    Display this help message.

    -v, --verbose
    Log the name of every file created and type generated to stdout.

    -n, --dry-run
    Do not write generated files to disk. Implies --verbose.

    -T, --types <type name or wildcard>
    Specify a type name or a wildcard pattern to generate code for. This option can be specified multiple times. If not specified, all types in the given .rdb files are generated.

    -X, --extra-types <.rdb file>
    Use an .rdb file containing types to be taken into account without generating output for them. This option can be specified multiple times.

    -Ocpp <directory>
    Specify the directory to write generated C++ files to.

    -Orust <directory>
    Specify the directory to write generated Rust files to.

    -O, --output-dir <directory> (legacy mode)
    Specify the directory to write generated files to. Creates subdirectories for C++ and Rust files.

Examples:
    rustmaker --verbose -T com.acme.XSomething \
        -X types.rdb -Ocpp acme/cpp -Orust acme/src acmetypes.rdb

    rustmaker --dry-run -T com.acme.* -X types.rdb \
        -X offapi.rdb -Ocpp acme/cpp -Orust acme/src acmetypes.rdb

    rustmaker -X types.rdb -Ocpp acme/cpp -Orust acme/src \
        acmetypes.rdb moretypes.rdb

    # Generate opaque pointer bindings for all UNO types
    rustmaker -T com.sun.star.frame.XComponentLoader \
        -X offapi.rdb -Ocpp cpp/ -Orust src/generated/ offapi.rdb

    # Legacy mode (backward compatibility)
    rustmaker --verbose -T com.acme.XSomething \
        -X types.rdb -O acme/ acmetypes.rdb
)"_ostr;
}

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
