/*
 *  $Id: rank-filter.c 21898 2019-02-21 14:01:55Z yeti-dn $
 *  Copyright (C) 2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/datafield.h>
#include <libprocess/arithmetic.h>
#include <libprocess/elliptic.h>
#include <libprocess/filters.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include "preview.h"

#define RANKFILTER_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

typedef struct {
    gdouble percentile1;
    gdouble percentile2;
    gint size;
    gboolean both;
    gboolean difference;
    gboolean new_channel;
} RankFilterArgs;

typedef struct {
    RankFilterArgs *args;

    GtkObject *radius;
    GtkObject *size;
    GtkObject *percentile1;
    GtkObject *rank1;
    GtkObject *percentile2;
    GtkObject *rank2;
    GtkWidget *both;
    GtkWidget *difference;
    GtkWidget *new_channel;
    gboolean in_update;

    GwySIValueFormat *vfpix;
    gdouble pixelscale;
    gdouble percentscale;
} RankFilterControls;

static gboolean      module_register            (void);
static void          rank_filter                (GwyContainer *data,
                                                 GwyRunType run);
static void          add_new_field              (GwyContainer *data,
                                                 gint oldid,
                                                 GwyDataField *dfield,
                                                 const gchar *title);
static gboolean      rank_filter_dialogue       (RankFilterArgs *args,
                                                 GwyDataField *dfield);
static void          radius_changed             (GtkAdjustment *adj,
                                                 RankFilterControls *controls);
static void          size_changed               (GtkAdjustment *adj,
                                                 RankFilterControls *controls);
static void          percentile1_changed        (GtkAdjustment *adj,
                                                 RankFilterControls *controls);
static void          rank1_changed              (GtkAdjustment *adj,
                                                 RankFilterControls *controls);
static void          percentile2_changed        (GtkAdjustment *adj,
                                                 RankFilterControls *controls);
static void          rank2_changed              (GtkAdjustment *adj,
                                                 RankFilterControls *controls);
static void          both_changed               (GtkToggleButton *toggle,
                                                 RankFilterControls *controls);
static void          difference_changed         (GtkToggleButton *toggle,
                                                 RankFilterControls *controls);
static void          new_channel_changed        (GtkToggleButton *toggle,
                                                 RankFilterControls *controls);
static void          update_sensitivity         (RankFilterControls *controls);
static void          update_rank_ranges         (RankFilterControls *controls);
static void          rank_filter_dialogue_update(RankFilterControls *controls,
                                                 const RankFilterArgs *args);
static GwyDataField* rank_filter_one_field      (gint kres,
                                                 GwyDataField *dfield,
                                                 gint rank);
static void          rank_filter_sanitize_args  (RankFilterArgs *args);
static void          rank_filter_load_args      (GwyContainer *container,
                                                 RankFilterArgs *args);
static void          rank_filter_save_args      (GwyContainer *container,
                                                 RankFilterArgs *args);

static const RankFilterArgs rank_filter_defaults = {
    75.0, 25.0,
    20,
    FALSE, FALSE, TRUE,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("General k-th rank filter replacing data with k-th rank values from "
       "the neighborhood."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2019",
};

GWY_MODULE_QUERY2(module_info, rank_filter)

static gboolean
module_register(void)
{
    gwy_process_func_register("rank_filter",
                              (GwyProcessFunc)&rank_filter,
                              N_("/_Integral Transforms/_Rank Filter..."),
                              NULL,
                              RANKFILTER_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("K-th rank filter"));

    return TRUE;
}

static void
rank_filter(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield, *rankfield1, *rankfield2;
    GtkWindow *window;
    RankFilterArgs args;
    gint oldid, kres, n, rank;
    GQuark dquark;
    gchar *s;
    gboolean ok = TRUE;

    g_return_if_fail(run & RANKFILTER_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_KEY, &dquark,
                                     GWY_APP_DATA_FIELD_ID, &oldid,
                                     0);
    g_return_if_fail(dfield && dquark);

    rank_filter_load_args(gwy_app_settings_get(), &args);

    if (run == GWY_RUN_INTERACTIVE) {
        ok = rank_filter_dialogue(&args, dfield);
        rank_filter_save_args(gwy_app_settings_get(), &args);
    }

    if (!ok)
        return;

    kres = 2*args.size + 1;
    n = gwy_data_field_get_elliptic_area_size(kres, kres);

    window = gwy_app_find_window_for_channel(data, oldid);
    gwy_app_wait_start(window, _("Filtering..."));

    rank = GWY_ROUND(args.percentile1/100.0 * n);
    rankfield1 = rank_filter_one_field(kres, dfield, rank);
    if (!rankfield1) {
        gwy_app_wait_finish();
        return;
    }

    if (!args.both) {
        gwy_app_wait_finish();
        if (args.new_channel) {
            s = g_strdup_printf(_("Rank filtered (%.1f %%)"), args.percentile1);
            add_new_field(data, oldid, rankfield1, s);
            g_free(s);
        }
        else {
            gwy_app_undo_qcheckpointv(data, 1, &dquark);
            gwy_data_field_assign(dfield, rankfield1);
            g_object_unref(rankfield1);
            gwy_data_field_data_changed(dfield);
            gwy_app_channel_log_add_proc(data, oldid, oldid);
        }
        return;
    }

    rank = GWY_ROUND(args.percentile2/100.0 * n);
    rankfield2 = rank_filter_one_field(kres, dfield, rank);
    gwy_app_wait_finish();
    if (!rankfield2) {
        g_object_unref(rankfield1);
        return;
    }

    if (args.difference) {
        gwy_data_field_subtract_fields(rankfield1, rankfield1, rankfield2);
        g_object_unref(rankfield2);
        add_new_field(data, oldid, rankfield1, _("Rank difference"));
        return;
    }

    s = g_strdup_printf(_("Rank filtered (%.1f %%)"), args.percentile1);
    add_new_field(data, oldid, rankfield1, s);
    g_free(s);

    s = g_strdup_printf(_("Rank filtered (%.1f %%)"), args.percentile2);
    add_new_field(data, oldid, rankfield2, s);
    g_free(s);
}

static void
add_new_field(GwyContainer *data, gint oldid,
              GwyDataField *dfield, const gchar *title)
{
    gint newid;

    newid = gwy_app_data_browser_add_data_field(dfield, data, TRUE);
    gwy_app_sync_data_items(data, data, oldid, newid, FALSE,
                            GWY_DATA_ITEM_GRADIENT,
                            0);
    gwy_app_set_data_field_title(data, newid, title);
    gwy_app_channel_log_add(data, oldid, newid, NULL, NULL);
    g_object_unref(dfield);
}

static gboolean
rank_filter_dialogue(RankFilterArgs *args, GwyDataField *dfield)
{
    GtkWidget *dialog, *table, *spin;
    RankFilterControls controls;
    gint response, row;
    gdouble q, xr, yr;

    gwy_clear(&controls, 1);
    controls.args = args;

    controls.vfpix
        = gwy_data_field_get_value_format_xy(dfield,
                                             GWY_SI_UNIT_FORMAT_VFMARKUP, NULL);

    /* FIXME: this is bogus for non-square pixels anyway */
    xr = gwy_data_field_get_dx(dfield);
    yr = gwy_data_field_get_dy(dfield);
    controls.pixelscale = hypot(xr, yr)/controls.vfpix->magnitude;

    dialog = gtk_dialog_new_with_buttons(_("Rank Filter"), NULL, 0,
                                         _("_Reset"), RESPONSE_RESET,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);

    table = gtk_table_new(12, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table,
                       FALSE, FALSE, 4);
    row = 0;
    controls.in_update = TRUE;

    gtk_table_attach(GTK_TABLE(table), gwy_label_new_header(_("Kernel Size")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    q = controls.pixelscale;
    controls.radius = gtk_adjustment_new(q*args->size, q, 1024*q, q, 10*q, 0);
    spin = gwy_table_attach_adjbar(table, row, _("Real _radius:"),
                                   controls.vfpix->units,
                                   controls.radius, GWY_HSCALE_SQRT);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin),
                               controls.vfpix->precision);
    g_signal_connect(controls.radius, "value-changed",
                     G_CALLBACK(radius_changed), &controls);
    row++;

    controls.size = gtk_adjustment_new(args->size, 1, 1024, 1, 10, 0);
    spin = gwy_table_attach_adjbar(table, row, _("_Pixel radius:"), _("px"),
                                   controls.size, GWY_HSCALE_SQRT);
    g_signal_connect(controls.size, "value-changed",
                     G_CALLBACK(size_changed), &controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    gtk_table_attach(GTK_TABLE(table), gwy_label_new_header(_("Rank")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.percentile1 = gtk_adjustment_new(args->percentile1, 0.0, 100.0,
                                              0.001, 0.1, 0);
    spin = gwy_table_attach_adjbar(table, row, _("_Percentile:"), "%",
                                   controls.percentile1, GWY_HSCALE_LINEAR);
    g_signal_connect(controls.percentile1, "value-changed",
                     G_CALLBACK(percentile1_changed), &controls);
    row++;

    /* We will fix the range later. */
    controls.rank1 = gtk_adjustment_new(0.0, 0.0, 1024.0, 1.0, 10.0, 0);
    spin = gwy_table_attach_adjbar(table, row, _("Ra_nk:"), NULL,
                                   controls.rank1,
                                   GWY_HSCALE_LINEAR | GWY_HSCALE_SNAP);
    g_signal_connect(controls.rank1, "value-changed",
                     G_CALLBACK(rank1_changed), &controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.both
        = gtk_check_button_new_with_mnemonic(_("_Second filter"));
    gtk_table_attach(GTK_TABLE(table), controls.both,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.both), args->both);
    g_signal_connect(controls.both, "toggled",
                     G_CALLBACK(both_changed), &controls);
    row++;

    controls.percentile2 = gtk_adjustment_new(args->percentile2, 0.0, 100.0,
                                              0.001, 0.1, 0);
    spin = gwy_table_attach_adjbar(table, row, _("_Percentile:"), "%",
                                   controls.percentile2, GWY_HSCALE_LINEAR);
    g_signal_connect(controls.percentile2, "value-changed",
                     G_CALLBACK(percentile2_changed), &controls);
    row++;

    /* We will fix the range later. */
    controls.rank2 = gtk_adjustment_new(0.0, 0.0, 1024.0, 1.0, 10.0, 0);
    spin = gwy_table_attach_adjbar(table, row, _("Ra_nk:"), NULL,
                                   controls.rank2,
                                   GWY_HSCALE_LINEAR | GWY_HSCALE_SNAP);
    g_signal_connect(controls.rank2, "value-changed",
                     G_CALLBACK(rank2_changed), &controls);
    row++;

    controls.difference
        = gtk_check_button_new_with_mnemonic(_("Calculate _difference"));
    gtk_table_attach(GTK_TABLE(table), controls.difference,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.difference),
                                 args->difference);
    g_signal_connect(controls.difference, "toggled",
                     G_CALLBACK(difference_changed), &controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    gtk_table_attach(GTK_TABLE(table), gwy_label_new_header(_("Options")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.new_channel
        = gtk_check_button_new_with_mnemonic(_("Create new image"));
    gtk_table_attach(GTK_TABLE(table), controls.new_channel,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.new_channel),
                                 args->new_channel);
    g_signal_connect(controls.new_channel, "toggled",
                     G_CALLBACK(new_channel_changed), &controls);
    row++;

    controls.in_update = FALSE;
    update_rank_ranges(&controls);
    update_sensitivity(&controls);

    gtk_widget_show_all(dialog);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            gwy_si_unit_value_format_free(controls.vfpix);
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            rank_filter_dialogue_update(&controls, &rank_filter_defaults);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);
    gwy_si_unit_value_format_free(controls.vfpix);

    return TRUE;
}

static void
radius_changed(GtkAdjustment *adj, RankFilterControls *controls)
{
    RankFilterArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->size = GWY_ROUND(gtk_adjustment_get_value(adj)/controls->pixelscale);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->size), args->size);
    controls->in_update = FALSE;
    update_rank_ranges(controls);
}

static void
size_changed(GtkAdjustment *adj, RankFilterControls *controls)
{
    RankFilterArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->size = gwy_adjustment_get_int(adj);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->radius),
                             args->size*controls->pixelscale);
    controls->in_update = FALSE;
    update_rank_ranges(controls);
}

static void
percentile1_changed(GtkAdjustment *adj, RankFilterControls *controls)
{
    RankFilterArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->percentile1 = gtk_adjustment_get_value(adj);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->rank1),
                             args->percentile1/controls->percentscale);
    controls->in_update = FALSE;
}

static void
rank1_changed(GtkAdjustment *adj, RankFilterControls *controls)
{
    RankFilterArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->percentile1 = gwy_adjustment_get_int(adj)*controls->percentscale;
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->percentile1),
                             args->percentile1);
    controls->in_update = FALSE;
}

static void
percentile2_changed(GtkAdjustment *adj, RankFilterControls *controls)
{
    RankFilterArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->percentile2 = gtk_adjustment_get_value(adj);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->rank2),
                             args->percentile2/controls->percentscale);
    controls->in_update = FALSE;
}

static void
rank2_changed(GtkAdjustment *adj, RankFilterControls *controls)
{
    RankFilterArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->percentile2 = gwy_adjustment_get_int(adj)*controls->percentscale;
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->percentile2),
                             args->percentile2);
    controls->in_update = FALSE;
}

static void
both_changed(GtkToggleButton *toggle, RankFilterControls *controls)
{
    RankFilterArgs *args = controls->args;

    args->both = gtk_toggle_button_get_active(toggle);
    update_sensitivity(controls);
}

static void
difference_changed(GtkToggleButton *toggle, RankFilterControls *controls)
{
    RankFilterArgs *args = controls->args;

    args->difference = gtk_toggle_button_get_active(toggle);
}

static void
new_channel_changed(GtkToggleButton *toggle, RankFilterControls *controls)
{
    RankFilterArgs *args = controls->args;

    args->new_channel = gtk_toggle_button_get_active(toggle);
}

static void
update_sensitivity(RankFilterControls *controls)
{
    RankFilterArgs *args = controls->args;

    gtk_widget_set_sensitive(controls->new_channel, !args->both);
    gwy_table_hscale_set_sensitive(controls->percentile2, args->both);
    gwy_table_hscale_set_sensitive(controls->rank2, args->both);
    gtk_widget_set_sensitive(controls->difference, args->both);
}

static void
rank_filter_dialogue_update(RankFilterControls *controls,
                            const RankFilterArgs *args)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->size), args->size);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->percentile1),
                             args->percentile1);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->percentile2),
                             args->percentile2);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->both), args->both);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->difference),
                                 args->difference);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->new_channel),
                                 args->new_channel);
}

static void
update_rank_ranges(RankFilterControls *controls)
{
    RankFilterArgs *args = controls->args;
    gint kres = 2*args->size + 1;
    gint n = gwy_data_field_get_elliptic_area_size(kres, kres);
    gdouble p1 = args->percentile1, p2 = args->percentile2;

    controls->percentscale = 100.0/n;
    controls->in_update = TRUE;
    g_object_set(controls->rank1, "upper", 1.0*n, NULL);
    g_object_set(controls->rank2, "upper", 1.0*n, NULL);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->rank1),
                             GWY_ROUND(p1/controls->percentscale));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->rank2),
                             GWY_ROUND(p2/controls->percentscale));
    controls->in_update = FALSE;
}

static GwyDataField*
rank_filter_one_field(gint kres, GwyDataField *dfield, gint rank)
{
    GwyDataField *rfield, *kernel;
    gboolean ok;

    kernel = gwy_data_field_new(kres, kres, 1.0, 1.0, TRUE);
    gwy_data_field_elliptic_area_fill(kernel, 0, 0, kres, kres, 1.0);
    rfield = gwy_data_field_duplicate(dfield);
    ok = gwy_data_field_area_filter_kth_rank(rfield, kernel, 0, 0,
                                             gwy_data_field_get_xres(rfield),
                                             gwy_data_field_get_yres(rfield),
                                             rank,
                                             gwy_app_wait_set_fraction);
    g_object_unref(kernel);
    if (!ok)
        GWY_OBJECT_UNREF(rfield);

    return rfield;
}

static const gchar both_key[]        = "/module/rank-filter/both";
static const gchar difference_key[]  = "/module/rank-filter/difference";
static const gchar new_channel_key[] = "/module/rank-filter/new_channel";
static const gchar percentile1_key[] = "/module/rank-filter/percentile1";
static const gchar percentile2_key[] = "/module/rank-filter/percentile2";
static const gchar size_key[]        = "/module/rank-filter/size";

static void
rank_filter_sanitize_args(RankFilterArgs *args)
{
    args->percentile1 = CLAMP(args->percentile1, 0.0, 100.0);
    args->percentile2 = CLAMP(args->percentile2, 0.0, 100.0);
    args->size = CLAMP(args->size, 1, 1024);
    args->both = !!args->both;
    args->difference = !!args->difference;
    args->new_channel = !!args->new_channel;
    if (args->both)
        args->new_channel = FALSE;
}

static void
rank_filter_load_args(GwyContainer *container,
                      RankFilterArgs *args)
{
    *args = rank_filter_defaults;

    gwy_container_gis_int32_by_name(container, size_key, &args->size);
    gwy_container_gis_double_by_name(container, percentile1_key,
                                     &args->percentile1);
    gwy_container_gis_double_by_name(container, percentile2_key,
                                     &args->percentile2);
    gwy_container_gis_boolean_by_name(container, both_key, &args->both);
    gwy_container_gis_boolean_by_name(container, difference_key,
                                      &args->difference);
    gwy_container_gis_boolean_by_name(container, new_channel_key,
                                      &args->new_channel);
    rank_filter_sanitize_args(args);
}

static void
rank_filter_save_args(GwyContainer *container,
                      RankFilterArgs *args)
{
    gwy_container_set_int32_by_name(container, size_key, args->size);
    gwy_container_set_double_by_name(container, percentile1_key,
                                     args->percentile1);
    gwy_container_set_double_by_name(container, percentile2_key,
                                     args->percentile2);
    gwy_container_set_boolean_by_name(container, both_key, args->both);
    gwy_container_set_boolean_by_name(container, difference_key,
                                      args->difference);
    gwy_container_set_boolean_by_name(container, new_channel_key,
                                      args->new_channel);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
