"""
A manual registry of renderers.

Renderers are the glue between a core and some output.  A service is the
combination of a subset of renderers and a core.

Renderers are actually defined in web.grend, but we need some way to
get at them from svcs and above, so the registry is kept here.
"""

#c Copyright 2008-2022, the GAVO project <gavo@ari.uni-heidelberg.de>
#c
#c This program is free software, covered by the GNU GPL.  See the
#c COPYING file in the source distribution.


import functools

from gavo import base
from gavo import utils


RENDERER_REGISTRY = {
	'admin': ("web.adminrender", "AdminRenderer"),
	'api': ("web.vodal", "APIRenderer"),
	'async': ("web.asyncrender", "DALIAsyncRenderer"),
	'availability': ("web.vosi", "VOSIAvailabilityRenderer"),
	'capabilities': ("web.vosi", "VOSICapabilityRenderer"),
	'coverage': ("web.metarender", "CoverageRenderer"),
	'custom': ("web.customrender", "CustomRenderer"),
	'dali': ("web.syncrender", "DALIRenderer"),
	'dlget': ("web.vodal", "DatalinkGetDataRenderer"),
	'dlasync': ("web.asyncrender", "DatalinkAsyncRenderer"),
	'dlmeta': ("web.vodal", "DatalinkGetMetaRenderer"),
	'edition': ("web.metarender", "EditionRenderer"),
	'examples': ("web.examplesrender", "Examples"),
	'external': ("web.metarender", "ExternalRenderer"),
	'fixed': ("web.constantrender", "FixedPageRenderer"),
	'form': ("web.formrender", "FormRenderer"),
	'get': ("web.productrender", "ProductRenderer"),
	'howtocite': ("web.metarender", "HowToCiteRenderer"),
	'info': ("web.metarender", "ServiceInfoRenderer"),
	'mupload': ("web.uploadservice", "MachineUploader"),
	'pubreg.xml': ("web.vodal", "RegistryRenderer"),
	'qp': ("web.qprenderer", "QPRenderer"),
	'rdinfo': ("web.metarender", "RDInfoRenderer"),
	'scs.xml': ("web.vodal", "SCSRenderer"),
	'siap.xml': ("web.vodal", "SIAPRenderer"),
	'siap2.xml': ("web.vodal", "SIAP2Renderer"),
	'slap.xml': ("web.vodal", "SLAPRenderer"),
	'soap': ("web.soaprender", "SOAPRenderer"),
	'ssap.xml': ("web.vodal", "SSAPRenderer"),
	'static': ("web.constantrender", "StaticRenderer"),
	'sync': ("web.syncrender", "SyncRenderer"),
	'tableinfo': ("web.metarender", "TableInfoRenderer"),
	'tableMetadata': ("web.vosi", "VOSITablesetRenderer"),
	'tables': ("web.vosi", "VOSITablesetRenderer"),
	'tablenote': ("web.metarender", "TableNoteRenderer"),
	'upload': ("web.uploadservice", "Uploader"),
	'uws.xml': ("web.asyncrender", "DALIAsyncRenderer"),
	'volatile': ("web.constantrender", "VolatilePageRenderer"),
}


@functools.lru_cache()
def getRenderer(name):
	if name not in RENDERER_REGISTRY:
		raise base.NotFoundError(name, "renderer", "registered renderers")
	cls = utils.loadInternalObject(*RENDERER_REGISTRY[name])

	if cls.name!=name and not name in cls.aliases:
		raise base.ReportableError("Internal Error: Renderer %s is registered"
			" under the wrong name."%name,
			hint="This is probably a typo in svcs.renderers; it needs"
			" to be fixed there")
	return cls

