﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/pinpoint/PinpointRequest.h>
#include <aws/pinpoint/Pinpoint_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Pinpoint {
namespace Model {

/**
 */
class GetSegmentVersionRequest : public PinpointRequest {
 public:
  AWS_PINPOINT_API GetSegmentVersionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetSegmentVersion"; }

  AWS_PINPOINT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique identifier for the application. This identifier is displayed as
   * the <b>Project ID</b> on the Amazon Pinpoint console.</p>
   */
  inline const Aws::String& GetApplicationId() const { return m_applicationId; }
  inline bool ApplicationIdHasBeenSet() const { return m_applicationIdHasBeenSet; }
  template <typename ApplicationIdT = Aws::String>
  void SetApplicationId(ApplicationIdT&& value) {
    m_applicationIdHasBeenSet = true;
    m_applicationId = std::forward<ApplicationIdT>(value);
  }
  template <typename ApplicationIdT = Aws::String>
  GetSegmentVersionRequest& WithApplicationId(ApplicationIdT&& value) {
    SetApplicationId(std::forward<ApplicationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier for the segment.</p>
   */
  inline const Aws::String& GetSegmentId() const { return m_segmentId; }
  inline bool SegmentIdHasBeenSet() const { return m_segmentIdHasBeenSet; }
  template <typename SegmentIdT = Aws::String>
  void SetSegmentId(SegmentIdT&& value) {
    m_segmentIdHasBeenSet = true;
    m_segmentId = std::forward<SegmentIdT>(value);
  }
  template <typename SegmentIdT = Aws::String>
  GetSegmentVersionRequest& WithSegmentId(SegmentIdT&& value) {
    SetSegmentId(std::forward<SegmentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique version number (Version property) for the campaign version.</p>
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  GetSegmentVersionRequest& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationId;

  Aws::String m_segmentId;

  Aws::String m_version;
  bool m_applicationIdHasBeenSet = false;
  bool m_segmentIdHasBeenSet = false;
  bool m_versionHasBeenSet = false;
};

}  // namespace Model
}  // namespace Pinpoint
}  // namespace Aws
