﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/workdocs/WorkDocsRequest.h>
#include <aws/workdocs/WorkDocs_EXPORTS.h>
#include <aws/workdocs/model/StorageRuleType.h>

#include <utility>

namespace Aws {
namespace WorkDocs {
namespace Model {

/**
 */
class CreateUserRequest : public WorkDocsRequest {
 public:
  AWS_WORKDOCS_API CreateUserRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateUser"; }

  AWS_WORKDOCS_API Aws::String SerializePayload() const override;

  AWS_WORKDOCS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the organization.</p>
   */
  inline const Aws::String& GetOrganizationId() const { return m_organizationId; }
  inline bool OrganizationIdHasBeenSet() const { return m_organizationIdHasBeenSet; }
  template <typename OrganizationIdT = Aws::String>
  void SetOrganizationId(OrganizationIdT&& value) {
    m_organizationIdHasBeenSet = true;
    m_organizationId = std::forward<OrganizationIdT>(value);
  }
  template <typename OrganizationIdT = Aws::String>
  CreateUserRequest& WithOrganizationId(OrganizationIdT&& value) {
    SetOrganizationId(std::forward<OrganizationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The login name of the user.</p>
   */
  inline const Aws::String& GetUsername() const { return m_username; }
  inline bool UsernameHasBeenSet() const { return m_usernameHasBeenSet; }
  template <typename UsernameT = Aws::String>
  void SetUsername(UsernameT&& value) {
    m_usernameHasBeenSet = true;
    m_username = std::forward<UsernameT>(value);
  }
  template <typename UsernameT = Aws::String>
  CreateUserRequest& WithUsername(UsernameT&& value) {
    SetUsername(std::forward<UsernameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The email address of the user.</p>
   */
  inline const Aws::String& GetEmailAddress() const { return m_emailAddress; }
  inline bool EmailAddressHasBeenSet() const { return m_emailAddressHasBeenSet; }
  template <typename EmailAddressT = Aws::String>
  void SetEmailAddress(EmailAddressT&& value) {
    m_emailAddressHasBeenSet = true;
    m_emailAddress = std::forward<EmailAddressT>(value);
  }
  template <typename EmailAddressT = Aws::String>
  CreateUserRequest& WithEmailAddress(EmailAddressT&& value) {
    SetEmailAddress(std::forward<EmailAddressT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The given name of the user.</p>
   */
  inline const Aws::String& GetGivenName() const { return m_givenName; }
  inline bool GivenNameHasBeenSet() const { return m_givenNameHasBeenSet; }
  template <typename GivenNameT = Aws::String>
  void SetGivenName(GivenNameT&& value) {
    m_givenNameHasBeenSet = true;
    m_givenName = std::forward<GivenNameT>(value);
  }
  template <typename GivenNameT = Aws::String>
  CreateUserRequest& WithGivenName(GivenNameT&& value) {
    SetGivenName(std::forward<GivenNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The surname of the user.</p>
   */
  inline const Aws::String& GetSurname() const { return m_surname; }
  inline bool SurnameHasBeenSet() const { return m_surnameHasBeenSet; }
  template <typename SurnameT = Aws::String>
  void SetSurname(SurnameT&& value) {
    m_surnameHasBeenSet = true;
    m_surname = std::forward<SurnameT>(value);
  }
  template <typename SurnameT = Aws::String>
  CreateUserRequest& WithSurname(SurnameT&& value) {
    SetSurname(std::forward<SurnameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The password of the user.</p>
   */
  inline const Aws::String& GetPassword() const { return m_password; }
  inline bool PasswordHasBeenSet() const { return m_passwordHasBeenSet; }
  template <typename PasswordT = Aws::String>
  void SetPassword(PasswordT&& value) {
    m_passwordHasBeenSet = true;
    m_password = std::forward<PasswordT>(value);
  }
  template <typename PasswordT = Aws::String>
  CreateUserRequest& WithPassword(PasswordT&& value) {
    SetPassword(std::forward<PasswordT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time zone ID of the user.</p>
   */
  inline const Aws::String& GetTimeZoneId() const { return m_timeZoneId; }
  inline bool TimeZoneIdHasBeenSet() const { return m_timeZoneIdHasBeenSet; }
  template <typename TimeZoneIdT = Aws::String>
  void SetTimeZoneId(TimeZoneIdT&& value) {
    m_timeZoneIdHasBeenSet = true;
    m_timeZoneId = std::forward<TimeZoneIdT>(value);
  }
  template <typename TimeZoneIdT = Aws::String>
  CreateUserRequest& WithTimeZoneId(TimeZoneIdT&& value) {
    SetTimeZoneId(std::forward<TimeZoneIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The amount of storage for the user.</p>
   */
  inline const StorageRuleType& GetStorageRule() const { return m_storageRule; }
  inline bool StorageRuleHasBeenSet() const { return m_storageRuleHasBeenSet; }
  template <typename StorageRuleT = StorageRuleType>
  void SetStorageRule(StorageRuleT&& value) {
    m_storageRuleHasBeenSet = true;
    m_storageRule = std::forward<StorageRuleT>(value);
  }
  template <typename StorageRuleT = StorageRuleType>
  CreateUserRequest& WithStorageRule(StorageRuleT&& value) {
    SetStorageRule(std::forward<StorageRuleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Amazon WorkDocs authentication token. Not required when using Amazon Web
   * Services administrator credentials to access the API.</p>
   */
  inline const Aws::String& GetAuthenticationToken() const { return m_authenticationToken; }
  inline bool AuthenticationTokenHasBeenSet() const { return m_authenticationTokenHasBeenSet; }
  template <typename AuthenticationTokenT = Aws::String>
  void SetAuthenticationToken(AuthenticationTokenT&& value) {
    m_authenticationTokenHasBeenSet = true;
    m_authenticationToken = std::forward<AuthenticationTokenT>(value);
  }
  template <typename AuthenticationTokenT = Aws::String>
  CreateUserRequest& WithAuthenticationToken(AuthenticationTokenT&& value) {
    SetAuthenticationToken(std::forward<AuthenticationTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_organizationId;

  Aws::String m_username;

  Aws::String m_emailAddress;

  Aws::String m_givenName;

  Aws::String m_surname;

  Aws::String m_password;

  Aws::String m_timeZoneId;

  StorageRuleType m_storageRule;

  Aws::String m_authenticationToken;
  bool m_organizationIdHasBeenSet = false;
  bool m_usernameHasBeenSet = false;
  bool m_emailAddressHasBeenSet = false;
  bool m_givenNameHasBeenSet = false;
  bool m_surnameHasBeenSet = false;
  bool m_passwordHasBeenSet = false;
  bool m_timeZoneIdHasBeenSet = false;
  bool m_storageRuleHasBeenSet = false;
  bool m_authenticationTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkDocs
}  // namespace Aws
